package JSci.awt;

import java.awt.*;

/**
* A bar graph AWT component.
* @version 1.0
* @author Mark Hale
*/
public class BarGraph extends CategoryGraph2D {
        /**
        * Line colors.
        */
        private Color lineColor[]={Color.black,Color.blue,Color.green,Color.red,Color.yellow,Color.cyan,Color.lightGray,Color.magenta,Color.orange,Color.pink};
        /**
        * Origin.
        */
        private Point origin;
        /**
        * Min and max data points.
        */
        private float minX,minY,maxX,maxY;
        /**
        * Axis scaling.
        */
        private float xScale,yScale;
        /**
        * Padding.
        */
        private final int scalePad=5,axisPad=50;
        /**
        * Axis numbering.
        */
        private boolean numbering=true;
        /**
        * Constructs a bar graph.
        */
        public BarGraph(CategoryGraph2DModel cgm) {
                super(cgm);
                dataChanged(new GraphDataEvent(model));
        }
        /**
        * Implementation of GraphDataListener.
        */
        public void dataChanged(GraphDataEvent e) {
                float tmp;
                model.firstSeries();
                minY=maxY=0.0f;
                minX=0.0f;maxX=model.seriesLength();
                for(int i=0;i<model.seriesLength();i++) {
                        tmp=model.getValue(i);
                        if(tmp>maxY)
                                maxY=tmp;
                        else if(tmp<minY)
                                minY=tmp;
                }
                while(model.nextSeries()) {
                        for(int i=0;i<model.seriesLength();i++) {
                                tmp=model.getValue(i);
                                if(tmp>maxY)
                                        maxY=tmp;
                                else if(tmp<minY)
                                        minY=tmp;
                        }
                }
                if(maxY==minY)
                        maxY=1.0f;
                rescale();
        }
        /**
        * Turns axis numbering on/off.
        */
        public void setNumbering(boolean flag) {
                numbering=flag;
        }
        /**
        * Returns the preferred size of this component.
        */
        public Dimension getPreferredSize() {
                return getMinimumSize();
        }
        /**
        * Returns the minimum size of this component.
        */
        public Dimension getMinimumSize() {
                return new Dimension(250,250);
        }
        /**
        * Paint the graph.
        */
        protected void offscreenPaint(Graphics g) {
// axis
                g.setColor(Color.black);
                g.drawLine(axisPad,origin.y,getSize().width-axisPad,origin.y);
                g.drawLine(origin.x,axisPad,origin.x,getSize().height-axisPad);
// bars
                Point p;
                int curSeries=0;
                model.firstSeries();
                g.setColor(lineColor[curSeries]);
                for(int i=0;i<model.seriesLength();i++) {
                        if(model.getValue(i)<0.0f)
                                p=dataToScreen(i,0.0f);
                        else
                                p=dataToScreen(i,model.getValue(i));
                        g.drawRect(p.x,p.y,Math.round(xScale),Math.abs(Math.round(yScale*model.getValue(i))));
                }
                while(model.nextSeries()) {
                        curSeries++;
                        g.setColor(lineColor[curSeries]);
                        for(int i=0;i<model.seriesLength();i++) {
                                if(model.getValue(i)<0.0f)
                                        p=dataToScreen(i,0.0f);
                                else
                                        p=dataToScreen(i,model.getValue(i));
                                g.drawRect(p.x,p.y,Math.round(xScale),Math.abs(Math.round(yScale*model.getValue(i))));
                        }
                }
                g.setColor(Color.black);
// x-axis labels
                int strWidth;
                final int strHeight=g.getFontMetrics().getHeight();
                for(int x=0;x<model.seriesLength();x++) {
                        strWidth=g.getFontMetrics().stringWidth(model.getCategory(x));
                        g.drawString(model.getCategory(x),dataToScreen(x+0.5f-0.5f*strWidth/xScale,0.0f).x,origin.y+strHeight);
                }
// numbering
                if(numbering) {
// y-axis numbering
                        final float dy=round(40.0f/yScale);
                        float y;
                        String yStr;
                        for(y=dy;y<=maxY;y+=dy) {
                                yStr=String.valueOf(round(y));
                                strWidth=g.getFontMetrics().stringWidth(yStr);
                                p=dataToScreen(0.0f,y);
                                g.drawLine(p.x,p.y,p.x-5,p.y);
                                g.drawString(yStr,origin.x-strWidth-8,p.y+strHeight/4);
                        }
                        for(y=-dy;y>=minY;y-=dy) {
                                yStr=String.valueOf(round(y));
                                strWidth=g.getFontMetrics().stringWidth(yStr);
                                p=dataToScreen(0.0f,y);
                                g.drawLine(p.x,p.y,p.x-5,p.y);
                                g.drawString(yStr,origin.x-strWidth-8,p.y+strHeight/4);
                        }
                }
        }
        /**
        * Reshapes the BarGraph to the specified bounding box.
        */
        public final void setBounds(int x,int y,int width,int height) {
                super.setBounds(x,y,width,height);
                rescale();
        }
        /**
        * Rescales the BarGraph.
        */
        private final void rescale() {
                final Dimension s=getMinimumSize();
                final int thisWidth=Math.max(getSize().width,s.width);
                final int thisHeight=Math.max(getSize().height,s.height);
                xScale=(thisWidth-2*(axisPad+scalePad))/(maxX-minX);
                yScale=(thisHeight-2*(axisPad+scalePad))/(maxY-minY);
                origin=new Point(Math.round(-minX*xScale)+axisPad+scalePad,Math.round(maxY*yScale)+axisPad+scalePad);
                redraw();
        }
        /**
        * Converts a data point to screen coordinates.
        */
        private final Point dataToScreen(float x,float y) {
                return new Point(origin.x+Math.round(xScale*x),origin.y-Math.round(yScale*y));
        }
}

