package JSci.maths;

import JSci.GlobalSettings;
import JSci.maths.fields.*;
import JSci.maths.groups.AbelianGroupMember;

/**
* The Complex class encapsulates complex numbers.
* @version 2.2
* @author Mark Hale
*/
public final class Complex extends Object implements FieldMember, java.io.Serializable {
        private double re;
        private double im;
        /**
        * Caching.
        */
        private boolean isModCached=false;
        private double modCache;
        private boolean isArgCached=false;
        private double argCache;
        /**
        * The complex number 0+1i.
        */
        public final static Complex I=ComplexField.I;
        /**
        * The complex number 1+0i.
        */
        public final static Complex ONE=ComplexField.ONE;
        /**
        * The complex number 0+0i.
        */
        public final static Complex ZERO=ComplexField.ZERO;
        /**
        * Constructs the complex number x+iy.
        * @param x the real value of a complex number
        * @param y the imaginary value of a complex number
        */
        public Complex(final double x,final double y) {
                re=x;
                im=y;
        }
        /**
        * Constructs the complex number represented by a string.
        * @param s a string representing a complex number
        * @exception NumberFormatException if the string does not contain a parsable number.
        */
        public Complex(final String s) throws NumberFormatException {
                final int iPos=s.indexOf('i');
                if(iPos==-1) {
                        re=Double.valueOf(s).doubleValue();
                        im=0.0;
                } else {
                        String imStr;
                        int signPos=s.indexOf('+',1);
                        if(signPos==-1)
                                signPos=s.indexOf('-',1);
                        if(signPos==-1) {
                                re=0.0;
                                imStr=s;
                        } else {
                                if(iPos<signPos) {
                                        imStr=s.substring(0,signPos);
                                        re=Double.valueOf(s.substring(signPos+1)).doubleValue();
                                } else {
                                        re=Double.valueOf(s.substring(0,signPos)).doubleValue();
                                        imStr=s.substring(signPos+1);
                                }
                        }
                        if(imStr.startsWith("i"))
                                im=Double.valueOf(imStr.substring(1)).doubleValue();
                        else if(imStr.endsWith("i"))
                                im=Double.valueOf(imStr.substring(0,imStr.length()-1)).doubleValue();
                        else
                                throw new NumberFormatException("The imaginary part should have 'i' as a prefix or suffix.");
                }
        }
        /**
        * Creates a complex number with the given modulus and argument.
        * @param mod the modulus of a complex number
        * @param arg the argument of a complex number
        */
        public static Complex polar(final double mod,final double arg) {
                final Complex z=new Complex(mod*Math.cos(arg),mod*Math.sin(arg));
                z.modCache=mod;
                z.isModCached=true;
                z.argCache=arg;
                z.isArgCached=true;
                return z;
        }
        /**
        * Compares two complex numbers for equality.
        * @param obj a complex number
        */
        public boolean equals(Object obj) {
                if(obj!=null && (obj instanceof Complex)) {
                        final Complex z=(Complex)obj;
                        return Math.abs(re-z.re)<=GlobalSettings.ZERO_TOL &&
                                Math.abs(im-z.im)<=GlobalSettings.ZERO_TOL;
                } else
                        return false;
        }
        /**
        * Returns a string representing the value of this complex number.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer();
                buf.append(re);
                if(im>=0.0)
                        buf.append("+");
                buf.append(im);
                buf.append("i");
                return buf.toString();
        }
        /**
        * Returns a hashcode for this complex number.
        */
        public int hashCode() {
                return (int)(Math.exp(mod()));
        }
        /**
        * Returns true if either the real or imaginary part is NaN.
        */
        public boolean isNaN() {
                return (re==Double.NaN) || (im==Double.NaN);
        }
        /**
        * Returns true if either the real or imaginary part is infinite.
        */
        public boolean isInfinite() {
                return (re==Double.POSITIVE_INFINITY) || (re==Double.NEGATIVE_INFINITY)
                        || (im==Double.POSITIVE_INFINITY) || (im==Double.NEGATIVE_INFINITY);
        }
        /**
        * Returns the real part of this complex number.
        */
        public double real() {
                return re;
        }
        /**
        * Returns the imaginary part of this complex number.
        */
        public double imag() {
                return im;
        }
        /**
        * Returns the modulus of this complex number.
        */
        public double mod() {
                if(isModCached)
                        return modCache;
                final double reAbs=Math.abs(re);
                final double imAbs=Math.abs(im);
                if(reAbs<=GlobalSettings.ZERO_TOL && imAbs<=GlobalSettings.ZERO_TOL) {
                        modCache=0.0;
                } else {
                        if(reAbs<imAbs)
                                modCache=imAbs*Math.sqrt(1.0+(re/im)*(re/im));
                        else
                                modCache=reAbs*Math.sqrt(1.0+(im/re)*(im/re));
                }
                isModCached=true;
                return modCache;
        }
        /**
        * Returns the argument of this complex number.
        */
        public double arg() {
                if(isArgCached)
                        return argCache;
                if(re>GlobalSettings.ZERO_TOL) {
                        argCache=Math.atan(im/re);
                } else if(re<-GlobalSettings.ZERO_TOL) {
                        if(im>=-GlobalSettings.ZERO_TOL)
                                argCache=Math.atan(im/re)+Math.PI;
                        else
                                argCache=Math.atan(im/re)-Math.PI;
                } else {
                        if(im>GlobalSettings.ZERO_TOL)
                                argCache=Math.PI/2.0;
                        else if(im<-GlobalSettings.ZERO_TOL)
                                argCache=-Math.PI/2.0;
                        else
                                argCache=0.0;
                }
                isArgCached=true;
                return argCache;
        }

//============
// OPERATIONS
//============

        /**
        * Returns the negative of this complex number.
        */
        public AbelianGroupMember negate() {
                return new Complex(-re,-im);
        }
        /**
        * Returns the inverse of this complex number.
        */
        public FieldMember inverse() {
                double denominator,real,imag;             
                if(Math.abs(re)<Math.abs(im)) {
                        real=re/im;
                        imag=-1.0;
                        denominator=re*real+im;
                } else {
                        real=1.0;
                        imag=-im/re;
                        denominator=re-im*imag;
                }
                return new Complex(real/denominator,imag/denominator);
        }
        /**
        * Returns the complex conjugate of this complex number.
        */
        public Complex conjugate() {
                return new Complex(re,-im);
        }

// ADDITION

        /**
        * Returns the addition of this number and another.
        */
        public AbelianGroupMember add(final AbelianGroupMember x) {
                if(x instanceof Complex)
                        return add((Complex)x);
                else if(x instanceof MathDouble)
                        return addReal(((MathDouble)x).value());
                else if(x instanceof MathInteger)
                        return addReal(((MathInteger)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the addition of this complex number and another.
        * @param z a complex number
        */
        public Complex add(final Complex z) {
                return new Complex(re+z.re,im+z.im);
        }
        /**
        * Returns the addition of this complex number with a real part.
        * @param real a real part
        */
        public Complex addReal(final double real) {
                return new Complex(re+real,im);
        }
        /**
        * Returns the addition of this complex number with an imaginary part.
        * @param imag an imaginary part
        */
        public Complex addImag(final double imag) {
                return new Complex(re,im+imag);
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this number and another.
        */
        public AbelianGroupMember subtract(final AbelianGroupMember x) {
                if(x instanceof Complex)
                        return subtract((Complex)x);
                else if(x instanceof MathDouble)
                        return subtractReal(((MathDouble)x).value());
                else if(x instanceof MathInteger)
                        return subtractReal(((MathInteger)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the subtraction of this complex number by another.
        * @param z a complex number
        */
        public Complex subtract(final Complex z) {
                return new Complex(re-z.re,im-z.im);
        }
        /**
        * Returns the subtraction of this complex number by a real part.
        * @param real a real part
        */
        public Complex subtractReal(final double real) {
                return new Complex(re-real,im);
        }
        /**
        * Returns the subtraction of this complex number by an imaginary part.
        * @param imag an imaginary part
        */
        public Complex subtractImag(final double imag) {
                return new Complex(re,im-imag);
        }

// MULTIPLICATION

        /**
        * Returns the multiplication of this number and another.
        */
        public RingMember multiply(final RingMember x) {
                if(x instanceof Complex)
                        return multiply((Complex)x);
                else if(x instanceof MathDouble)
                        return multiply(((MathDouble)x).value());
                else if(x instanceof MathInteger)
                        return multiply(((MathInteger)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the multiplication of this complex number and another.
        * @param z a complex number
        */
        public Complex multiply(final Complex z) {
                return new Complex(
                        re*z.re-im*z.im,
                        re*z.im+im*z.re
                );
        }
        /**
        * Returns the multiplication of this complex number by a scalar.
        * @param x a real number
        */
        public Complex multiply(final double x) {
                return new Complex(x*re,x*im);
        }

// DIVISION

        /**
        * Returns the division of this number and another.
        */
        public FieldMember divide(final FieldMember x) {
                if(x instanceof Complex)
                        return divide((Complex)x);
                else if(x instanceof MathDouble)
                        return divide(((MathDouble)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the division of this complex number by another.
        * @param z a complex number
        * @exception ArithmeticException If divide by zero.
        */
        public Complex divide(final Complex z) {
                double denominator,real,imag,a;             
                if(Math.abs(z.re)<Math.abs(z.im)) {
                        a=z.re/z.im;
                        denominator=z.re*a+z.im;
                        real=re*a+im;
                        imag=im*a-re;
                } else {
                        a=z.im/z.re;
                        denominator=z.re+z.im*a;
                        real=re+im*a;
                        imag=im-re*a;
                }
                return new Complex(real/denominator,imag/denominator);
        }
        /**
        * Returns the division of this complex number by a scalar.
        * @param x a real number
        * @exception ArithmeticException If divide by zero.
        */
        public Complex divide(final double x) {
                return new Complex(re/x,im/x);
        }

// POWER

        /**
        * Returns this complex number raised to the power of another.
        * @param z a complex number
        */
        public Complex pow(final Complex z) {
                final double thisMod=mod();
                final double thisArg=arg();
                final double r=Math.pow(thisMod, z.re)/Math.exp(thisArg*z.im);
                final double a=thisArg*z.re+Math.log(thisMod)*z.im;
                return polar(r,a);
        }
        /**
        * Returns this complex number raised to the power of a scalar.
        * @param x a real number
        */
        public Complex pow(final double x) {
                return polar(Math.pow(mod(),x),arg()*x);
        }
        /**
        * Returns the square of this complex number.
        */
        public Complex sqr() {
                return new Complex(re*re-im*im,2.0*re*im);
        }
        /**
        * Returns the square root of this complex number.
        */
        public Complex sqrt() {
                return polar(Math.pow(mod(),0.5),arg()*0.5);
        }

//===========
// FUNCTIONS
//===========

// EXP

        /**
        * Returns the exponential number e(2.718...) raised to the power of a complex number.
        * @param z a complex number
        */
        public static Complex exp(final Complex z) {
                return new Complex(
                        Math.exp(z.re)*Math.cos(z.im),
                        Math.exp(z.re)*Math.sin(z.im)
                );
        }

// LOG

        /**
        * Returns the natural logarithm (base e) of a complex number.
        * @param z a complex number
        */
        public static Complex log(final Complex z) {
                return new Complex(Math.log(z.mod()),z.arg());
        }

// SIN

        /**
        * Returns the trigonometric sine of a complex angle.
        * @param z an angle that is measured in radians
        */
        public static Complex sin(final Complex z) {
                return new Complex(
                        Math.sin(z.re)*ExtraMath.cosh(z.im),
                        Math.cos(z.re)*ExtraMath.sinh(z.im)
                );
        }

// COS

        /**
        * Returns the trigonometric cosine of a complex angle.
        * @param z an angle that is measured in radians
        */
        public static Complex cos(final Complex z) {
                return new Complex(
                        Math.cos(z.re)*ExtraMath.cosh(z.im),
                       -Math.sin(z.re)*ExtraMath.sinh(z.im)
                );
        }

// TAN

        /**
        * Returns the trigonometric tangent of a complex angle.
        * @param z an angle that is measured in radians
        */
        public static Complex tan(final Complex z) {
                final double sinRe=Math.sin(z.re);
                final double cosRe=Math.cos(z.re);
                final double sinhIm=ExtraMath.sinh(z.im);
                final double coshIm=ExtraMath.cosh(z.im);
                final double denom=cosRe*cosRe*coshIm*coshIm+sinRe*sinRe*sinhIm*sinhIm;
                return new Complex(sinRe*cosRe/denom,sinhIm*coshIm/denom);
        }

// SINH

        /**
        * Returns the hyperbolic sine of a complex number.
        * @param z a complex number
        */
        public static Complex sinh(final Complex z) {
                return new Complex(
                        ExtraMath.sinh(z.re)*Math.cos(z.im),
                        ExtraMath.cosh(z.re)*Math.sin(z.im)
                );
        }

// COSH

        /**
        * Returns the hyperbolic cosine of a complex number.
        * @param z a complex number
        */
        public static Complex cosh(final Complex z) {
                return new Complex(
                        ExtraMath.cosh(z.re)*Math.cos(z.im),
                        ExtraMath.sinh(z.re)*Math.sin(z.im)
                );
        }

// TANH

        /**
        * Returns the hyperbolic tangent of a complex number.
        * @param z a complex number
        */
        public static Complex tanh(final Complex z) {
                final double sinhRe=ExtraMath.sinh(z.re);
                final double coshRe=ExtraMath.cosh(z.re);
                final double sinIm=Math.sin(z.im);
                final double cosIm=Math.cos(z.im);
                final double denom=coshRe*coshRe*cosIm*cosIm+sinhRe*sinhRe*sinIm*sinIm;
                return new Complex(sinhRe*coshRe/denom,sinIm*cosIm/denom);
        }

// INVERSE SIN

        /**
        * Returns the arc sine of a complex number, in the range of
        * (-<I>pi</I>/2 through <I>pi</I>/2, -<I>infinity</I> through <I>infinity</I>).
        * @param z a complex number
        */
        public static Complex asin(final Complex z) {
                if(z.equals(ONE))
                        return ComplexField.PI_2;
                else if(z.equals(ComplexField.MINUS_ONE))
                        return ComplexField.MINUS_PI_2;
                else {
                        // atan(z/sqrt(1-z*z))
                        final double oneMzzRe=1.0-z.re*z.re+z.im*z.im;
                        final double oneMzzIm=-2.0*z.re*z.im;
                        return atan(z.divide((new Complex(oneMzzRe,oneMzzIm)).sqrt()));
                }
        }

// INVERSE COS

        /**
        * Returns the arc cosine of a complex number, in the range of
        * (0.0 through <I>pi</I>, 0.0 through <I>infinity</I>).
        * @param z a complex number
        */
        public static Complex acos(final Complex z) {
                if(z.equals(ONE))
                        return ZERO;
                else if(z.equals(ComplexField.MINUS_ONE))
                        return ComplexField.PI;
                else {
                        // atan(-z/sqrt(1-z*z))+PI/2
                        final double oneMzzRe=1.0-z.re*z.re+z.im*z.im;
                        final double oneMzzIm=-2.0*z.re*z.im;
                        return ComplexField.PI_2.subtract(atan(z.divide((new Complex(oneMzzRe,oneMzzIm)).sqrt())));
                }
        }

// INVERSE TAN

        /**
        * Returns the arc tangent of a complex number, in the range of
        * (-<I>pi</I>/2 through <I>pi</I>/2, -<I>infinity</I> through <I>infinity</I>).
        * @param z a complex number
        */
        public static Complex atan(final Complex z) {
                // -i atanh(iz)
                return ComplexField.MINUS_I.multiply(atanh(I.multiply(z)));
        }

// INVERSE SINH

        /**
        * Returns the arc hyperbolic sine of a complex number, in the range of
        * (-<I>infinity</I> through <I>infinity</I>, -<I>pi</I>/2 through <I>pi</I>/2).
        * @param z a complex number
        */
        public static Complex asinh(final Complex z) {
                if(z.equals(I))
                        return ComplexField.PI_2_I;
                else if(z.equals(ComplexField.MINUS_I))
                        return ComplexField.MINUS_PI_2_I;
                else {
                        // log(z+sqrt(z*z+1))
                        final double zzPoneRe=z.re*z.re-z.im*z.im+1.0;
                        final double zzPoneIm=2.0*z.re*z.im;
                        return log(z.add((new Complex(zzPoneRe,zzPoneIm)).sqrt()));
                }
        }

// INVERSE COSH

        /**
        * Returns the arc hyperbolic cosine of a complex number, in the range of
        * (0.0 through <I>infinity</I>, 0.0 through <I>pi</I>).
        * @param z a complex number
        */
        public static Complex acosh(final Complex z) {
                if(z.equals(ONE))
                        return ZERO;
                else if(z.equals(ComplexField.MINUS_ONE))
                        return ComplexField.PI_I;
                else {
                        // log(z+sqrt(z*z-1))
                        final double zzMoneRe=z.re*z.re-z.im*z.im-1.0;
                        final double zzMoneIm=2.0*z.re*z.im;
                        return log(z.add((new Complex(zzMoneRe,zzMoneIm)).sqrt()));
                }
        }

// INVERSE TANH

        /**
        * Returns the arc hyperbolic tangent of a complex number, in the range of
        * (-<I>infinity</I> through <I>infinity</I>, -<I>pi</I>/2 through <I>pi</I>/2).
        * @param z a complex number
        */
        public static Complex atanh(final Complex z) {
                // 1/2 log((1+z)/(1-z))
                final double denom=1.0-2.0*z.re+z.im*z.im+z.re*z.re;
                return log(new Complex((1.0-z.re*z.re-z.im*z.im)/denom,2.0*z.im/denom)).divide(2.0);
        }
}

