package JSci.maths;

/**
* The Complex3Vector class encapsulates complex 3-vectors.
* @version 2.0
* @author Mark Hale
*/
public class Complex3Vector extends ComplexVector {
        /**
        * Constructs an empty 3-vector.
        */
        public Complex3Vector() {
                super(3);
        }
        /**
        * Constructs a 3-vector.
        * @param x x coordinate
        * @param y y coordinate
        * @param z z coordinate
        */
        public Complex3Vector(final Complex x,final Complex y,final Complex z) {
                super(3);
                vector[0]=x;
                vector[1]=y;
                vector[2]=z;
        }
        /**
        * Compares two complex vectors for equality.
        * @param a a complex 3-vector
        */
        public boolean equals(Object a) {
                if(a!=null && (a instanceof Complex3Vector)) {
                        final Complex3Vector cv=(Complex3Vector)a;
                        return (vector[0].equals(cv.vector[0]) && vector[1].equals(cv.vector[1]) &&
                                vector[2].equals(cv.vector[2]));
                } else
                        return false;
        }
        /**
        * Returns a comma delimited string representing the value of this vector.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer(5);
                buf.append(vector[0]).append(',').append(vector[1]).append(',').append(vector[2]);
                return buf.toString();
        }
        /**
        * Returns a hashcode for this vector.
        */
        public int hashCode() {
                return (int)Math.exp(norm());
        }
        /**
        * Returns the real part of this complex 3-vector.
        */
        public DoubleVector real() {
                return new Double3Vector(vector[0].real(),vector[1].real(),vector[2].real());
        }
        /**
        * Returns the imaginary part of this complex 3-vector.
        */
        public DoubleVector imag() {
                return new Double3Vector(vector[0].imag(),vector[1].imag(),vector[2].imag());
        }
        /**
        * Returns a component of this vector.
        * @param n index of the vector component
        * @exception VectorDimensionException If attempting to access an invalid component.
        */
        public Complex getComponent(final int n) {
                if(n>=0 && n<3)
                        return vector[n];
                else
                        throw new VectorDimensionException("Invalid component.");
        }
        /**
        * Sets the value of a component of this vector.
        * @param n index of the vector component
        * @param z a complex number
        * @exception VectorDimensionException If attempting to access an invalid component.
        */
        public void setComponent(final int n, final Complex z) {
                if(n>=0 && n<3)
                        vector[n]=z;
                else
                        throw new VectorDimensionException("Invalid component.");
        }
        /**
        * Returns the vector's dimension.
        */
        public int dimension() {
                return 3;
        }
        /**
        * Returns the l2-norm (magnitude).
        */
        public double norm() {
                return Math.sqrt(vector[0].real()*vector[0].real()+vector[0].imag()*vector[0].imag()
                +vector[1].real()*vector[1].real()+vector[1].imag()*vector[1].imag()
                +vector[2].real()*vector[2].real()+vector[2].imag()*vector[2].imag());
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public double infNorm() {
                double infNorm=vector[0].mod();
                if(vector[1].mod()>infNorm)
                        infNorm=vector[1].mod();
                if(vector[2].mod()>infNorm)
                        infNorm=vector[2].mod();
                return infNorm;
        }

//============
// OPERATIONS
//============

// COMPLEX CONJUGATE

        /**
        * Returns the complex conjugate of this vector.
        * @return a complex 3-vector
        */
        public ComplexVector conjugate() {
                return new Complex3Vector(vector[0].conjugate(),vector[1].conjugate(),vector[2].conjugate());
        }

// ADDITION

        /**
        * Returns the addition of this vector and another.
        */
        public MathVector add(final MathVector v) {
                if(v instanceof Complex3Vector)
                        return add((Complex3Vector)v);
                else if(v instanceof Double3Vector)
                        return add((Double3Vector)v);
                else if(v instanceof Integer3Vector)
                        return add((Integer3Vector)v);
                else
                        throw new IllegalArgumentException("Vector class not recognised by this method.");
        }
        /**
        * Returns the addition of this vector and another.
        * @param v a complex 3-vector
        */
        public Complex3Vector add(final Complex3Vector v) {
                return new Complex3Vector(
                        vector[0].add(v.vector[0]),
                        vector[1].add(v.vector[1]),
                        vector[2].add(v.vector[2])
                );
        }
        /**
        * Returns the addition of this vector and another.
        * @param v a double 3-vector
        */
        public Complex3Vector add(final Double3Vector v) {
                return new Complex3Vector(
                        vector[0].addReal(v.vector[0]),
                        vector[1].addReal(v.vector[1]),
                        vector[2].addReal(v.vector[2])
                );
        }
        /**
        * Returns the addition of this vector and another.
        * @param v an integer 3-vector
        */
        public Complex3Vector add(final Integer3Vector v) {
                return new Complex3Vector(
                        vector[0].addReal(v.vector[0]),
                        vector[1].addReal(v.vector[1]),
                        vector[2].addReal(v.vector[2])
                );
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this vector by another.
        */
        public MathVector subtract(final MathVector v) {
                if(v instanceof Complex3Vector)
                        return subtract((Complex3Vector)v);
                else if(v instanceof Double3Vector)
                        return subtract((Double3Vector)v);
                else if(v instanceof Integer3Vector)
                        return subtract((Integer3Vector)v);
                else
                        throw new IllegalArgumentException("Vector class not recognised by this method.");
        }
        /**
        * Returns the subtraction of this vector by another.
        * @param v a complex 3-vector
        */
        public Complex3Vector subtract(final Complex3Vector v) {
                return new Complex3Vector(
                        vector[0].subtract(v.vector[0]),
                        vector[1].subtract(v.vector[1]),
                        vector[2].subtract(v.vector[2])
                );
        }
        /**
        * Returns the subtraction of this vector by another.
        * @param v a double 3-vector
        */
        public Complex3Vector subtract(final Double3Vector v) {
                return new Complex3Vector(
                        vector[0].subtractReal(v.vector[0]),
                        vector[1].subtractReal(v.vector[1]),
                        vector[2].subtractReal(v.vector[2])
                );
        }
        /**
        * Returns the subtraction of this vector by another.
        * @param v an integer 3-vector
        */
        public Complex3Vector subtract(final Integer3Vector v) {
                return new Complex3Vector(
                        vector[0].subtractReal(v.vector[0]),
                        vector[1].subtractReal(v.vector[1]),
                        vector[2].subtractReal(v.vector[2])
                );
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this vector by a scalar.
        * @param z a complex number
        * @return a complex 3-vector
        */
        public ComplexVector scalarMultiply(final Complex z) {
                return new Complex3Vector(vector[0].multiply(z),vector[1].multiply(z),vector[2].multiply(z));
        }
        /**
        * Returns the multiplication of this vector by a scalar.
        * @param x a double
        * @return a complex 3-vector
        */
        public ComplexVector scalarMultiply(final double x) {
                return new Complex3Vector(vector[0].multiply(x),vector[1].multiply(x),vector[2].multiply(x));
        }

// SCALAR DIVISION

        /**
        * Returns the division of this vector by a scalar.
        * @param z a complex number
        * @return a complex 3-vector
        * @exception ArithmeticException If divide by zero.
        */
        public ComplexVector scalarDivide(final Complex z) {
                return new Complex3Vector(vector[0].divide(z),vector[1].divide(z),vector[2].divide(z));
        }
        /**
        * Returns the division of this vector by a scalar.
        * @param x a double
        * @return a complex 3-vector
        * @exception ArithmeticException If divide by zero.
        */
        public ComplexVector scalarDivide(final double x) {
                return new Complex3Vector(vector[0].divide(x),vector[1].divide(x),vector[2].divide(x));
        }

// SCALAR PRODUCT

        /**
        * Returns the scalar product of this vector and another.
        * @param v a complex 3-vector
        */
        public Complex scalarProduct(final Complex3Vector v) {
                return new Complex(
                        vector[0].real()*v.vector[0].real()+vector[0].imag()*v.vector[0].imag()
                        +vector[1].real()*v.vector[1].real()+vector[1].imag()*v.vector[1].imag()
                        +vector[2].real()*v.vector[2].real()+vector[2].imag()*v.vector[2].imag(),
                        vector[0].imag()*v.vector[0].real()-vector[0].real()*v.vector[0].imag()
                        +vector[1].imag()*v.vector[1].real()-vector[1].real()*v.vector[1].imag()
                        +vector[2].imag()*v.vector[2].real()-vector[2].real()*v.vector[2].imag()
                );
        }

// VECTOR PRODUCT

        /**
        * Returns the vector product of this vector and another.
        * @param v a complex 3-vector
        */
        public Complex3Vector multiply(final Complex3Vector v) {
                return new Complex3Vector(
                        (vector[1].multiply(v.vector[2])).subtract(v.vector[1].multiply(vector[2])),
                        (vector[2].multiply(v.vector[0])).subtract(v.vector[2].multiply(vector[0])),
                        (vector[0].multiply(v.vector[1])).subtract(v.vector[0].multiply(vector[1]))
                );
        }

// MAP COMPONENTS

        /**
        * Applies a function on all the vector components.
        * @param f a user-defined function
        * @return a complex 3-vector
        */
        public ComplexVector mapComponents(final Mapping f) {
                return new Complex3Vector(f.map(vector[0]),f.map(vector[1]),f.map(vector[2]));
        }
}

