package JSci.maths;

/**
* The ComplexDiagonalMatrix class provides an object for encapsulating diagonal matrices containing complex numbers.
* Uses compressed diagonal storage.
* @version 2.1
* @author Mark Hale
*/
public final class ComplexDiagonalMatrix extends ComplexTridiagonalMatrix {
        /**
        * Storage format identifier.
        */
        protected final static int DIAGONAL=4;
        protected final static int storageFormat=DIAGONAL;
        /**
        * Constructs a matrix.
        */
        protected ComplexDiagonalMatrix() {
                super();
        }
        /**
        * Constructs an empty matrix.
        * @param size the number of rows/columns
        */
        public ComplexDiagonalMatrix(final int size) {
                this();
                matrix=new Complex[1][size];
        }
        /**
        * Constructs a matrix from an array.
        * @param array an assigned value
        * @exception MatrixDimensionException If the array is not square.
        */
        public ComplexDiagonalMatrix(final Complex array[][]) {
                this(array.length);
                if(array.length==array[0].length) {
                        for(int i=0;i<matrix[0].length;i++)
                                matrix[0][i]=array[i][i];
                } else {
                        matrix=null;
                        throw new MatrixDimensionException("Array must be square.");
                }
        }
        /**
        * Constructs a matrix by wrapping an array containing the diagonal elements.
        * @param array an assigned value
        */
        public ComplexDiagonalMatrix(Complex array[]) {
                this();
                matrix=new Complex[1][];
                matrix[0]=array;
        }
        /**
        * Creates an identity matrix.
        * @param size the number of rows/columns
        */
        public static ComplexDiagonalMatrix identity(final int size) {
                final Complex array[]=new Complex[size];
                for(int i=0;i<size;i++)
                        array[i]=Complex.ONE;
                return new ComplexDiagonalMatrix(array);
        }
        /**
        * Compares two complex diagonal matrices for equality.
        * @param m a complex diagonal matrix
        */
        public boolean equals(Object m) {
                if(m!=null && (m instanceof ComplexDiagonalMatrix) &&
                matrix[0].length==((ComplexDiagonalMatrix)m).rows()) {
                        final ComplexDiagonalMatrix cdm=(ComplexDiagonalMatrix)m;
                        for(int i=0;i<matrix[0].length;i++) {
                                if(!matrix[0][i].equals(cdm.getElement(i,i)))
                                        return false;
                        }
                        return true;
                } else
                        return false;
        }
        /**
        * Returns the real part of this complex matrix.
        * @return a double diagonal matrix
        */
        public DoubleMatrix real() {
                final double array[]=new double[matrix[0].length];
                for(int i=0;i<matrix[0].length;i++)
                        array[i]=matrix[0][i].real();
                return new DoubleDiagonalMatrix(array);
        }
        /**
        * Returns the imaginary part of this complex matrix.
        * @return a double diagonal matrix
        */
        public DoubleMatrix imag() {
                final double array[]=new double[matrix[0].length];
                for(int i=0;i<matrix[0].length;i++)
                        array[i]=matrix[0][i].imag();
                return new DoubleDiagonalMatrix(array);
        }
        /**
        * Returns an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public Complex getElement(final int i, final int j) {
                if(i>=0 && i<matrix[0].length && j>=0 && j<matrix[0].length) {
                        if(i==j)
                                return matrix[0][i];
                        else
                                return Complex.ZERO;
                } else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Sets the value of an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @param z a complex number
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public void setElement(final int i, final int j, final Complex z) {
                if(i>=0 && i<matrix[0].length && j>=0 && j<matrix[0].length && i==j)
                        matrix[0][i]=z;
                else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Returns true if this matrix is hermitian.
        */
        public boolean isHermitian() {
                return this.equals(this.conjugate());
        }
        /**
        * Returns true if this matrix is unitary.
        */
        public boolean isUnitary() {
                return this.multiply(this.conjugate()).equals(identity(matrix[0].length));
        }
        /**
        * Returns the determinant.
        */
        public Complex det() {
                Complex det=matrix[0][0];
                for(int i=1;i<matrix[0].length;i++)
                        det=det.multiply(matrix[0][i]);
                return det;
        }
        /**
        * Returns the trace.
        */
        public Complex trace() {
                double trRe=matrix[0][0].real();
                double trIm=matrix[0][0].imag();
                for(int i=1;i<matrix[0].length;i++) {
                        trRe+=matrix[0][i].real();
                        trIm+=matrix[0][i].imag();
                }
                return new Complex(trRe,trIm);
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public double infNorm() {
                double result=matrix[0][0].mod();
                double tmpResult;
                for(int i=1;i<matrix[0].length;i++) {
                        tmpResult=matrix[0][i].mod();
                        if(tmpResult>result)
                                result=tmpResult;
                }
                return result;
        }
        /**
        * Returns the Frobenius norm.
        * @author Taber Smith
        */
        public double frobeniusNorm() {
                double result=matrix[0][0].mod()*matrix[0][0].mod();
                for(int i=1;i<matrix[0].length;i++)
                        result+=matrix[0][i].mod()*matrix[0][i].mod();
                return Math.sqrt(result);
        }
        /**
        * Returns the number of rows.
        */
        public int rows() {
                return matrix[0].length;
        }
        /**
        * Returns the number of columns.
        */
        public int columns() {
                return matrix[0].length;
        }

//============
// OPERATIONS
//============

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        * @param m a complex matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexMatrix add(final ComplexMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        case TRIDIAGONAL: return rawAdd((ComplexTridiagonalMatrix)m);
                        case DIAGONAL: return rawAdd((ComplexDiagonalMatrix)m);
                        default: 
                                if(matrix[0].length==m.rows() && matrix[0].length==m.columns()) {
                                        final Complex array[][]=new Complex[matrix[0].length][matrix[0].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=m.getElement(i,j);
                                        }
                                        for(int i=0;i<array.length;i++)
                                                array[i][i]=array[i][i].add(matrix[0][i]);
                                        return new ComplexSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private ComplexSquareMatrix rawAdd(final ComplexMatrix m) {
                if(matrix[0].length==m.matrix.length && matrix[0].length==m.matrix[0].length) {
                        final Complex array[][]=new Complex[matrix[0].length][matrix[0].length];
                        for(int i=0;i<array.length;i++)
                                System.arraycopy(m.matrix[i],0,array[i],0,array.length);
                        for(int i=0;i<array.length;i++)
                                array[i][i]=array[i][i].add(matrix[0][i]);
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a complex square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexSquareMatrix add(final ComplexSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        case TRIDIAGONAL: return rawAdd((ComplexTridiagonalMatrix)m);
                        case DIAGONAL: return rawAdd((ComplexDiagonalMatrix)m);
                        default: 
                                if(matrix[0].length==m.rows()) {
                                        final Complex array[][]=new Complex[matrix[0].length][matrix[0].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                       array[i][j]=m.getElement(i,j);
                                        }
                                        for(int i=0;i<array.length;i++)
                                                array[i][i]=array[i][i].add(matrix[0][i]);
                                        return new ComplexSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a complex tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexTridiagonalMatrix add(final ComplexTridiagonalMatrix m) {
                switch(m.storageFormat) {
                        case TRIDIAGONAL: return rawAdd(m);
                        case DIAGONAL: return rawAdd((ComplexDiagonalMatrix)m);
                        default: 
                                int mRow=matrix[0].length;
                                if(mRow==m.rows()) {
                                        final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(mRow);
                                        ans.matrix[1][0]=matrix[0][0].add(m.getElement(0,0));
                                        ans.matrix[2][0]=m.getElement(0,1);
                                        mRow--;
                                        for(int i=1;i<mRow;i++) {
                                                ans.matrix[0][i]=m.getElement(i,i-1);
                                                ans.matrix[1][i]=matrix[0][i].add(m.getElement(i,i));
                                                ans.matrix[2][i]=m.getElement(i,i+1);
                                        }
                                        ans.matrix[0][mRow]=m.getElement(mRow,mRow-1);
                                        ans.matrix[1][mRow]=matrix[0][mRow].add(m.getElement(mRow,mRow));
                                        return ans;
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private ComplexTridiagonalMatrix rawAdd(final ComplexTridiagonalMatrix m) {
                if(matrix[0].length==m.matrix[1].length) {
                        final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(matrix[0].length);
                        System.arraycopy(m.matrix[0],0,ans.matrix[0],0,matrix[0].length);
                        System.arraycopy(m.matrix[2],0,ans.matrix[2],0,matrix[2].length);
                        ans.matrix[1][0]=matrix[0][0].add(m.matrix[1][0]);
                        for(int i=1;i<matrix[0].length;i++)
                                ans.matrix[1][i]=matrix[0][i].add(m.matrix[1][i]);
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a complex diagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexDiagonalMatrix add(final ComplexDiagonalMatrix m) {
                if(matrix[0].length==m.matrix[0].length) {
                        final Complex array[]=new Complex[matrix[0].length];
                        array[0]=matrix[0][0].add(m.getElement(0,0));
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i].add(m.matrix[0][i]);
                        return new ComplexDiagonalMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix by another.
        */
        public Matrix subtract(final Matrix m) {
                if(m instanceof ComplexDiagonalMatrix)
                        return subtract((ComplexDiagonalMatrix)m);
                else if(m instanceof ComplexTridiagonalMatrix)
                        return rawSubtract((ComplexTridiagonalMatrix)m);
                else if(m instanceof ComplexSquareMatrix)
                        return rawSubtract((ComplexSquareMatrix)m);
                else if(m instanceof ComplexMatrix)
                        return rawSubtract((ComplexMatrix)m);
                else
                        throw new IllegalArgumentException("Matrix class not recognised by this method.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a complex matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexMatrix subtract(final ComplexMatrix m) {
                if(m instanceof ComplexDiagonalMatrix)
                        return subtract((ComplexDiagonalMatrix)m);
                else if(m instanceof ComplexTridiagonalMatrix)
                        return rawSubtract((ComplexTridiagonalMatrix)m);
                else if(m instanceof ComplexSquareMatrix)
                        return rawSubtract((ComplexSquareMatrix)m);
                else
                        return rawSubtract(m);
        }
        private ComplexMatrix rawSubtract(final ComplexMatrix m) {
                if(matrix[0].length==m.rows() && matrix[0].length==m.columns()) {
                        final Complex array[][]=new Complex[matrix[0].length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=(Complex)m.matrix[i][0].negate();
                                for(j=1;j<array.length;j++)
                                        array[i][j]=(Complex)m.matrix[i][j].negate();
                        }
                        for(int i=0;i<array.length;i++)
                                array[i][i]=array[i][i].add(matrix[0][i]);
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a complex square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexSquareMatrix subtract(final ComplexSquareMatrix m) {
                if(m instanceof ComplexDiagonalMatrix)
                        return subtract((ComplexDiagonalMatrix)m);
                else if(m instanceof ComplexTridiagonalMatrix)
                        return rawSubtract((ComplexTridiagonalMatrix)m);
                else
                        return rawSubtract(m);
        }
        private ComplexSquareMatrix rawSubtract(final ComplexSquareMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final Complex array[][]=new Complex[matrix[0].length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=(Complex)m.matrix[i][0].negate();
                                for(j=1;j<array.length;j++)
                                        array[i][j]=(Complex)m.matrix[i][j].negate();
                        }
                        for(int i=0;i<array.length;i++)
                                array[i][i]=array[i][i].add(matrix[0][i]);
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a complex tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexTridiagonalMatrix subtract(final ComplexTridiagonalMatrix m) {
                if(m instanceof ComplexDiagonalMatrix)
                        return subtract((ComplexDiagonalMatrix)m);
                else
                        return rawSubtract(m);
        }
        private ComplexTridiagonalMatrix rawSubtract(final ComplexTridiagonalMatrix m) {
                if(matrix[0].length==m.rows()) {
                        int mRow=matrix[0].length;
                        final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(mRow);
                        ans.setElement(0,0,matrix[0][0].subtract(m.getElement(0,0)));
                        ans.setElement(1,0,(Complex)m.getElement(1,0).negate());
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.setElement(i-1,i,(Complex)m.getElement(i-1,i).negate());
                                ans.setElement(i,i,matrix[0][i].subtract(m.getElement(i,i)));
                                ans.setElement(i+1,i,(Complex)m.getElement(i+1,i).negate());
                        }
                        ans.setElement(mRow-1,mRow,(Complex)m.getElement(mRow-1,mRow).negate());
                        ans.setElement(mRow,mRow,matrix[0][mRow].subtract(m.getElement(mRow,mRow)));
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a complex diagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexDiagonalMatrix subtract(final ComplexDiagonalMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final Complex array[]=new Complex[matrix[0].length];
                        array[0]=matrix[0][0].subtract(m.matrix[0][0]);
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i].subtract(m.matrix[0][i]);
                        return new ComplexDiagonalMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SCALAR MULTIPLY

        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param z a complex number
        * @return a complex diagonal matrix
        */
        public ComplexMatrix scalarMultiply(final Complex z) {
                final Complex array[]=new Complex[matrix[0].length];
                array[0]=z.multiply(matrix[0][0]);
                for(int i=1;i<array.length;i++)
                        array[i]=z.multiply(matrix[0][i]);
                return new ComplexDiagonalMatrix(array);
        }
        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x a double
        * @return a complex diagonal matrix
        */
        public ComplexMatrix scalarMultiply(final double x) {
                final Complex array[]=new Complex[matrix[0].length];
                array[0]=matrix[0][0].multiply(x);
                for(int i=1;i<array.length;i++)
                        array[i]=matrix[0][i].multiply(x);
                return new ComplexDiagonalMatrix(array);
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v a complex vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public ComplexVector multiply(final ComplexVector v) {
                if(matrix[0].length==v.dimension()) {
                        final Complex array[]=new Complex[matrix[0].length];
                        array[0]=matrix[0][0].multiply(v.getComponent(0));
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i].multiply(v.getComponent(i));
                        return new ComplexVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        */
        public Matrix multiply(final Matrix m) {
                if(m instanceof ComplexDiagonalMatrix)
                        return multiply((ComplexDiagonalMatrix)m);
                else if(m instanceof ComplexTridiagonalMatrix)
                        return rawMultiply((ComplexTridiagonalMatrix)m);
                else if(m instanceof ComplexSquareMatrix)
                        return rawMultiply((ComplexSquareMatrix)m);
                else if(m instanceof ComplexMatrix)
                        return rawMultiply((ComplexMatrix)m);
                else
                        throw new IllegalArgumentException("Matrix class not recognised by this method.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a complex matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexMatrix multiply(final ComplexMatrix m) {
                if(m instanceof ComplexDiagonalMatrix)
                        return multiply((ComplexDiagonalMatrix)m);
                else if(m instanceof ComplexTridiagonalMatrix)
                        return rawMultiply((ComplexTridiagonalMatrix)m);
                else if(m instanceof ComplexSquareMatrix)
                        return rawMultiply((ComplexSquareMatrix)m);
                else
                        return rawMultiply(m);
        }
        private ComplexMatrix rawMultiply(final ComplexMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final Complex array[][]=new Complex[matrix[0].length][m.columns()];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[0][i].multiply(m.getElement(i,0));
                                for(j=1;j<array[0].length;j++)
                                        array[i][j]=matrix[0][i].multiply(m.getElement(i,j));
                        }
                        return new ComplexMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a complex square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexSquareMatrix multiply(final ComplexSquareMatrix m) {
                if(m instanceof ComplexDiagonalMatrix)
                        return multiply((ComplexDiagonalMatrix)m);
                else if(m instanceof ComplexTridiagonalMatrix)
                        return rawMultiply((ComplexTridiagonalMatrix)m);
                else
                        return rawMultiply(m);
        }
        private ComplexSquareMatrix rawMultiply(final ComplexSquareMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final Complex array[][]=new Complex[matrix[0].length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[0][i].multiply(m.getElement(i,0));
                                for(j=1;j<array.length;j++)
                                        array[i][j]=matrix[0][i].multiply(m.getElement(i,j));
                        }
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a complex tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexSquareMatrix multiply(final ComplexTridiagonalMatrix m) {
                if(m instanceof ComplexDiagonalMatrix)
                        return multiply((ComplexDiagonalMatrix)m);
                else
                        return rawMultiply(m);
        }
        private ComplexSquareMatrix rawMultiply(final ComplexTridiagonalMatrix m) {
                int mRow=matrix[0].length;
                if(mRow==m.rows()) {
                        final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(mRow);
                        ans.setElement(0,0,matrix[0][0].multiply(m.getElement(0,0)));
                        ans.setElement(0,1,matrix[0][0].multiply(m.getElement(0,1)));
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.setElement(i,i-1,matrix[0][i].multiply(m.getElement(i,i-1)));
                                ans.setElement(i,i,matrix[0][i].multiply(m.getElement(i,i)));
                                ans.setElement(i,i+1,matrix[0][i].multiply(m.getElement(i,i+1)));
                        }
                        ans.setElement(mRow,mRow-1,matrix[0][mRow].multiply(m.getElement(mRow,mRow-1)));
                        ans.setElement(mRow,mRow,matrix[0][mRow].multiply(m.getElement(mRow,mRow)));
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a complex diagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexDiagonalMatrix multiply(final ComplexDiagonalMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final Complex array[]=new Complex[matrix[0].length];
                        array[0]=matrix[0][0].multiply(m.getElement(0,0));
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i].multiply(m.getElement(i,i));
                        return new ComplexDiagonalMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// INVERSE

        /**
        * Returns the inverse of this matrix.
        * @return a complex diagonal matrix
        */
        public ComplexSquareMatrix inverse() {
                final Complex array[]=new Complex[matrix[0].length];
                array[0]=Complex.ONE.divide(matrix[0][0]);
                for(int i=1;i<array.length;i++)
                        array[i]=Complex.ONE.divide(matrix[0][i]);
                return new ComplexDiagonalMatrix(array);
        }

// CONJUGATE

        /**
        * Returns the complex conjugate of this matrix.
        * @return a complex diagonal matrix
        */
        public ComplexMatrix conjugate() {
                final Complex array[]=new Complex[matrix[0].length];
                array[0]=matrix[0][0].conjugate();
                for(int i=1;i<array.length;i++)
                        array[i]=matrix[0][i].conjugate();
                return new ComplexDiagonalMatrix(array);
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return a complex diagonal matrix
        */
        public Matrix transpose() {
                return this;
        }

// LU DECOMPOSITION

        /**
        * Returns the LU decomposition of this matrix.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public ComplexSquareMatrix[] luDecompose() {
                final ComplexDiagonalMatrix lu[]=new ComplexDiagonalMatrix[2];
                lu[0]=identity(matrix[0].length);
                lu[1]=this;
                return lu;
        }

// MAP ELEMENTS

        /**
        * Applies a function on all the matrix elements.
        * @param f a user-defined function
        * @return a complex diagonal matrix
        */
        public ComplexMatrix mapElements(final Mapping f) {
                final Complex array[]=new Complex[matrix[0].length];
                array[0]=f.map(matrix[0][0]);
                for(int i=1;i<array.length;i++)
                        array[i]=f.map(matrix[0][i]);
                return new ComplexDiagonalMatrix(array);
        }
}

