package JSci.maths;

import JSci.maths.groups.AbelianGroupMember;

/**
* The ComplexMatrix class provides an object for encapsulating matrices containing complex numbers.
* @version 2.1
* @author Mark Hale
*/
public class ComplexMatrix extends Matrix {
        /**
        * Storage format identifier.
        */
        protected final static int ARRAY_2D=1;
        protected final static int storageFormat=ARRAY_2D;
        /**
        * Array containing the elements of the matrix.
        */
        protected Complex matrix[][];
        /**
        * Constructs a matrix.
        */
        protected ComplexMatrix() {}
        /**
        * Constructs an empty matrix.
        * @param row the number of rows
        * @param col the number of columns
        */
        public ComplexMatrix(final int row,final int col) {
                matrix=new Complex[row][col];
        }
        /**
        * Constructs a matrix by wrapping an array.
        * @param array an assigned value
        */
        public ComplexMatrix(final Complex array[][]) {
                matrix=array;
        }
        /**
        * Constructs a matrix from an array of vectors (columns).
        * @param array an assigned value
        */
        public ComplexMatrix(ComplexVector array[]) {
                this(array[0].dimension(),array.length);
                for(int j,i=0;i<matrix.length;i++) {
                        for(j=0;j<matrix[0].length;j++)
                                matrix[i][j]=array[j].getComponent(i);
                }
        }
        /**
        * Finalize.
        * @exception Throwable Any that occur.
        */
        protected void finalize() throws Throwable {
                matrix=null;
                super.finalize();
        }
        /**
        * Compares two complex matrices for equality.
        * @param m a complex matrix
        */
        public boolean equals(Object m) {
                if(m!=null && (m instanceof ComplexMatrix) &&
                matrix.length==((ComplexMatrix)m).rows() && matrix[0].length==((ComplexMatrix)m).columns()) {
                        final ComplexMatrix cm=(ComplexMatrix)m;
                        for(int j,i=0;i<matrix.length;i++) {
                                for(j=0;j<matrix[0].length;j++) {
                                        if(!matrix[i][j].equals(cm.getElement(i,j)))
                                                return false;
                                }
                        }
                        return true;
                } else {
                        return false;
                }
        }
        /**
        * Returns a string representing this matrix.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer(matrix.length*matrix[0].length);
                for(int j,i=0;i<matrix.length;i++) {
                        for(j=0;j<matrix[0].length;j++) {
                                buf.append(matrix[i][j].toString());
                                buf.append(' ');
                        }
                        buf.append('\n');
                }
                return buf.toString();
        }
        /**
        * Returns a hashcode for this matrix.
        */
        public int hashCode() {
                return (int)Math.exp(infNorm());
        }
        /**
        * Returns the real part of this complex matrix.
        * @return a double matrix
        */
        public DoubleMatrix real() {
                final double array[][]=new double[matrix.length][matrix[0].length];
                for(int j,i=0;i<matrix.length;i++) {
                        for(j=0;j<matrix[0].length;j++)
                                array[i][j]=matrix[i][j].real();
                }
                return new DoubleMatrix(array);
        }
        /**
        * Returns the imaginary part of this complex matrix.
        * @return a double matrix
        */
        public DoubleMatrix imag() {
                final double array[][]=new double[matrix.length][matrix[0].length];
                for(int j,i=0;i<matrix.length;i++) {
                        for(j=0;j<matrix[0].length;j++)
                                array[i][j]=matrix[i][j].imag();
                }
                return new DoubleMatrix(array);
        }
        /**
        * Returns an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public Complex getElement(final int i, final int j) {
                if(i>=0 && i<matrix.length && j>=0 && j<matrix[0].length)
                        return matrix[i][j];
                else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Sets the value of an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @param z a complex number
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public void setElement(final int i, final int j, final Complex z) {
                if(i>=0 && i<matrix.length && j>=0 && j<matrix[0].length)
                        matrix[i][j]=z;
                else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public double infNorm() {
                double result=0.0,tmpResult;
                for(int j,i=0;i<matrix.length;i++) {
                        tmpResult=0.0;
                        for(j=0;j<matrix[0].length;j++)
                                tmpResult+=matrix[i][j].mod();
                        if(tmpResult>result)
                                result=tmpResult;
                }
                return result;
        }
        /**
        * Returns the Frobenius norm.
        * @author Taber Smith
        */
        public double frobeniusNorm() {
                double result=0.0;
                for(int j,i=0;i<matrix.length;i++)
                        for(j=0;j<matrix[0].length;j++)
                                result+=matrix[i][j].mod()*matrix[i][j].mod();
                return Math.sqrt(result);
        }
        /**
        * Returns the number of rows.
        */
        public int rows() {
                return matrix.length;
        }
        /**
        * Returns the number of columns.
        */
        public int columns() {
                return matrix[0].length;
        }

//============
// OPERATIONS
//============

        /**
        * Returns the negative of this matrix.
        */
        public AbelianGroupMember negate() {
                final Complex array[][]=new Complex[matrix.length][matrix[0].length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=(Complex)matrix[i][0].negate();
                        for(j=1;j<array[0].length;j++)
                                array[i][j]=(Complex)matrix[i][j].negate();
                }
                return new ComplexMatrix(array);
        }

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        */
        public AbelianGroupMember add(final AbelianGroupMember m) {
                if(m instanceof ComplexMatrix)
                        return add((ComplexMatrix)m);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a complex matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexMatrix add(final ComplexMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        default: 
                                if(matrix.length==m.rows() && matrix[0].length==m.columns()) {
                                        final Complex array[][]=new Complex[matrix.length][matrix[0].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0].add(m.getElement(i,0));
                                                for(j=1;j<array[0].length;j++)
                                                        array[i][j]=matrix[i][j].add(m.getElement(i,j));
                                        }
                                        return new ComplexMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private ComplexMatrix rawAdd(final ComplexMatrix m) {
                if(matrix.length==m.matrix.length && matrix[0].length==m.matrix[0].length) {
                        final Complex array[][]=new Complex[matrix.length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[i][0].add(m.matrix[i][0]);
                                for(j=1;j<array[0].length;j++)
                                        array[i][j]=matrix[i][j].add(m.matrix[i][j]);
                        }
                        return new ComplexMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix by another.
        */
        public AbelianGroupMember subtract(final AbelianGroupMember m) {
                if(m instanceof ComplexMatrix)
                        return subtract((ComplexMatrix)m);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a complex matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexMatrix subtract(final ComplexMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        default: 
                                if(matrix.length==m.rows() && matrix[0].length==m.columns()) {
                                        final Complex array[][]=new Complex[matrix.length][matrix[0].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0].subtract(m.getElement(i,0));
                                                for(j=1;j<array[0].length;j++)
                                                        array[i][j]=matrix[i][j].subtract(m.getElement(i,j));
                                        }
                                        return new ComplexMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private ComplexMatrix rawSubtract(final ComplexMatrix m) {
                if(matrix.length==m.matrix.length && matrix[0].length==m.matrix[0].length) {
                        final Complex array[][]=new Complex[matrix.length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[i][0].subtract(m.matrix[i][0]);
                                for(j=1;j<array[0].length;j++)
                                        array[i][j]=matrix[i][j].subtract(m.matrix[i][j]);
                        }
                        return new ComplexMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this matrix by a scalar.
        */
        public AlgebraMember scalarMultiply(RingMember x) {
                if(x instanceof Complex)
                        return scalarMultiply((Complex)x);
                else if(x instanceof MathDouble)
                        return scalarMultiply(((MathDouble)x).value());
                else if(x instanceof MathInteger)
                        return scalarMultiply(((MathInteger)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param z a complex number
        * @return a complex matrix
        */
        public ComplexMatrix scalarMultiply(final Complex z) {
                final Complex array[][]=new Complex[matrix.length][matrix[0].length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=z.multiply(matrix[i][0]);
                        for(j=1;j<array[0].length;j++)
                                array[i][j]=z.multiply(matrix[i][j]);
                }
                return new ComplexMatrix(array);
        }
        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x a double
        * @return a complex matrix
        */
        public ComplexMatrix scalarMultiply(final double x) {
                final Complex array[][]=new Complex[matrix.length][matrix[0].length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=matrix[i][0].multiply(x);
                        for(j=1;j<array[0].length;j++)
                                array[i][j]=matrix[i][j].multiply(x);
                }
                return new ComplexMatrix(array);
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v a complex vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public ComplexVector multiply(final ComplexVector v) {
                if(matrix[0].length==v.dimension()) {
                        final Complex array[]=new Complex[matrix.length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i]=matrix[i][0].multiply(v.getComponent(0));
                                for(j=1;j<matrix[0].length;j++)
                                        array[i]=array[i].add(matrix[i][j].multiply(v.getComponent(j)));
                        }
                        return new ComplexVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        */
        public RingMember multiply(final RingMember m) {
                if(m instanceof ComplexMatrix)
                        return multiply((ComplexMatrix)m);
                else
                        throw new IllegalArgumentException("Matrix class not recognised by this method.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a complex matrix
        * @return a ComplexMatrix or a ComplexSquareMatrix as appropriate
        * @exception MatrixDimensionException If the matrices are incompatible.
        */
        public ComplexMatrix multiply(final ComplexMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        default: 
                                if(matrix[0].length==m.rows()) {
                                        int n,k;
                                        final Complex array[][]=new Complex[matrix.length][m.columns()];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array[0].length;k++) {
                                                        array[j][k]=matrix[j][0].multiply(m.getElement(0,k));
                                                        for(n=1;n<matrix[0].length;n++)
                                                                array[j][k]=array[j][k].add(matrix[j][n].multiply(m.getElement(n,k)));
                                                }
                                        }
                                        if(array.length==array[0].length)
                                                return new ComplexSquareMatrix(array);
                                        else
                                                return new ComplexMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private ComplexMatrix rawMultiply(final ComplexMatrix m) {
                if(matrix[0].length==m.matrix.length) {
                        int n,k;
                        final Complex array[][]=new Complex[matrix.length][m.matrix[0].length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array[0].length;k++) {
                                        array[j][k]=matrix[j][0].multiply(m.matrix[0][k]);
                                        for(n=1;n<matrix[0].length;n++)
                                                array[j][k]=array[j][k].add(matrix[j][n].multiply(m.matrix[n][k]));
                                }
                        }
                        if(array.length==array[0].length)
                                return new ComplexSquareMatrix(array);
                        else
                                return new ComplexMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }

// HERMITIAN ADJOINT

        /**
        * Returns the hermitian adjoint of this matrix.
        * @return a complex matrix
        */
        public ComplexMatrix hermitianAdjoint() {
                final Complex array[][]=new Complex[matrix[0].length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[0][i]=matrix[i][0].conjugate();
                        for(j=1;j<array[0].length;j++)
                                array[j][i]=matrix[i][j].conjugate();
                }
                return new ComplexMatrix(array);
        }

// CONJUGATE

        /**
        * Returns the complex conjugate of this matrix.
        * @return a complex matrix
        */
        public ComplexMatrix conjugate() {
                final Complex array[][]=new Complex[matrix.length][matrix[0].length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=matrix[i][0].conjugate();
                        for(j=1;j<array[0].length;j++)
                                array[i][j]=matrix[i][j].conjugate();
                }
                return new ComplexMatrix(array);
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return a complex matrix
        */
        public Matrix transpose() {
                final Complex array[][]=new Complex[matrix[0].length][matrix.length];
                for(int j,i=0;i<array[0].length;i++) {
                        array[0][i]=matrix[i][0];
                        for(j=1;j<array.length;j++)
                                array[j][i]=matrix[i][j];
                }
                return new ComplexMatrix(array);
        }

// MAP ELEMENTS

        /**
        * Applies a function on all the matrix elements.
        * @param f a user-defined function
        * @return a complex matrix
        */
        public ComplexMatrix mapElements(final Mapping f) {
                final Complex array[][]=new Complex[matrix.length][matrix[0].length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=f.map(matrix[i][0]);
                        for(j=1;j<array[0].length;j++)
                                array[i][j]=f.map(matrix[i][j]);
                }
                return new ComplexMatrix(array);
        }
}

