package JSci.maths;

/**
* The ComplexSquareMatrix class provides an object for encapsulating square matrices containing complex numbers.
* @version 2.1
* @author Mark Hale
*/
public class ComplexSquareMatrix extends ComplexMatrix {
        /**
        * Constructs a matrix.
        */
        protected ComplexSquareMatrix() {
                super();
        }
        /**
        * Constructs an empty matrix.
        * @param size the number of rows/columns
        */
        public ComplexSquareMatrix(final int size) {
                super(size,size);
        }
        /**
        * Constructs a matrix by wrapping an array.
        * @param array an assigned value
        * @exception MatrixDimensionException If the array is not square.
        */
        public ComplexSquareMatrix(final Complex array[][]) {
                super(array);
                if(array.length!=array[0].length) {
                        matrix=null;
                        throw new MatrixDimensionException("The array is not square.");
                }
        }
        /**
        * Constructs a matrix from an array of vectors (columns).
        * @param array an assigned value
        */
        public ComplexSquareMatrix(final ComplexVector array[]) {
                super(array);
                if(array.length!=array[0].dimension()) {
                        matrix=null;
                        throw new MatrixDimensionException("The array does not form a square matrix.");
                }
        }
        /**
        * Returns true if this matrix is hermitian.
        */
        public boolean isHermitian() {
                return this.equals(this.hermitianAdjoint());
        }
        /**
        * Returns true if this matrix is unitary.
        */
        public boolean isUnitary() {
                return this.multiply(this.hermitianAdjoint()).equals(ComplexDiagonalMatrix.identity(matrix[0].length));
        }
        /**
        * Returns the determinant.
        */
        public Complex det() {
                if(matrix.length==2) {
                        return (matrix[0][0].multiply(matrix[1][1])).subtract(matrix[0][1].multiply(matrix[1][0]));
                } else {
                        final ComplexSquareMatrix lu[]=this.luDecompose();
                        Complex det=lu[1].matrix[0][0];
                        for(int i=1;i<matrix.length;i++)
                                det=det.multiply(lu[1].matrix[i][i]);
                        return det;
                }
        }
        /**
        * Returns the trace.
        * @author Taber Smith
        */
        public Complex trace() {
                double trRe=matrix[0][0].real();
                double trIm=matrix[0][0].imag();
                for(int i=1;i<matrix.length;i++) {
                        trRe+=matrix[0][i].real();
                        trIm+=matrix[0][i].imag();
                }
                return new Complex(trRe,trIm);
        }

//============
// OPERATIONS
//============

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        * @param m a complex matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexMatrix add(final ComplexMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        default: 
                                if(matrix.length==m.rows() && matrix.length==m.columns()) {
                                        final Complex array[][]=new Complex[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0].add(m.getElement(i,0));
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j].add(m.getElement(i,j));
                                        }
                                        return new ComplexSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private ComplexSquareMatrix rawAdd(final ComplexMatrix m) {
                if(matrix.length==m.matrix.length && matrix.length==m.matrix[0].length) {
                        final Complex array[][]=new Complex[matrix.length][matrix.length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[i][0].add(m.matrix[i][0]);
                                for(j=1;j<array.length;j++)
                                        array[i][j]=matrix[i][j].add(m.matrix[i][j]);
                        }
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a complex square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexSquareMatrix add(final ComplexSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        default: 
                                if(matrix.length==m.rows()) {
                                        final Complex array[][]=new Complex[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0].subtract(m.getElement(i,0));
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j].subtract(m.getElement(i,j));
                                        }
                                        return new ComplexSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix by another.
        * @param m a complex matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexMatrix subtract(final ComplexMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        default: 
                                if(matrix.length==m.rows() && matrix.length==m.columns()) {
                                        final Complex array[][]=new Complex[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0].subtract(m.getElement(i,0));
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j].subtract(m.getElement(i,j));
                                        }
                                        return new ComplexSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private ComplexSquareMatrix rawSubtract(final ComplexMatrix m) {
                if(matrix.length==m.matrix.length && matrix.length==m.matrix[0].length) {
                        final Complex array[][]=new Complex[matrix.length][matrix.length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[i][0].subtract(m.matrix[i][0]);
                                for(j=1;j<array.length;j++)
                                        array[i][j]=matrix[i][j].subtract(m.matrix[i][j]);
                        }
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a complex square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexSquareMatrix subtract(final ComplexSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        default: 
                                if(matrix.length==m.rows()) {
                                        final Complex array[][]=new Complex[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0].subtract(m.getElement(i,0));
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j].subtract(m.getElement(i,j));
                                        }
                                        return new ComplexSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param z a complex number
        * @return a complex square matrix
        */
        public ComplexMatrix scalarMultiply(final Complex z) {
                final Complex array[][]=new Complex[matrix.length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=z.multiply(matrix[i][0]);
                        for(j=1;j<array.length;j++)
                                array[i][j]=z.multiply(matrix[i][j]);
                }
                return new ComplexSquareMatrix(array);
        }
        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x a double
        * @return a complex square matrix
        */
        public ComplexMatrix scalarMultiply(final double x) {
                final Complex array[][]=new Complex[matrix.length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=matrix[i][0].multiply(x);
                        for(j=1;j<array.length;j++)
                                array[i][j]=matrix[i][j].multiply(x);
                }
                return new ComplexSquareMatrix(array);
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v a complex vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public ComplexVector multiply(final ComplexVector v) {
                if(matrix.length==v.dimension()) {
                        final Complex array[]=new Complex[matrix.length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i]=matrix[i][0].multiply(v.getComponent(0));
                                for(j=1;j<matrix.length;j++)
                                        array[i]=array[i].add(matrix[i][j].multiply(v.getComponent(j)));
                        }
                        return new ComplexVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a complex matrix
        * @exception MatrixDimensionException If the matrices are incompatible.
        */
        public ComplexMatrix multiply(final ComplexMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        default:
                                if(matrix[0].length==m.rows()) {
                                        int n,k;
                                        final Complex array[][]=new Complex[matrix.length][m.columns()];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array[0].length;k++) {
                                                        array[j][k]=matrix[j][0].multiply(m.getElement(0,k));
                                                        for(n=1;n<matrix[0].length;n++)
                                                                array[j][k]=array[j][k].add(matrix[j][n].multiply(m.getElement(n,k)));
                                                }
                                        }
                                        return new ComplexMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private ComplexMatrix rawMultiply(final ComplexMatrix m) {
                if(matrix[0].length==m.matrix.length) {
                        int n,k;
                        final Complex array[][]=new Complex[matrix.length][m.matrix[0].length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array[0].length;k++) {
                                        array[j][k]=matrix[j][0].multiply(m.matrix[0][k]);
                                        for(n=1;n<matrix[0].length;n++)
                                                array[j][k]=array[j][k].add(matrix[j][n].multiply(m.matrix[n][k]));
                                }
                        }
                        return new ComplexMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a complex square matrix
        * @exception MatrixDimensionException If the matrices are incompatible.
        */
        public ComplexSquareMatrix multiply(final ComplexSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        default:
                                if(matrix.length==m.rows()) {
                                        int n,k;
                                        final Complex array[][]=new Complex[matrix.length][matrix.length];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array.length;k++) {
                                                        array[j][k]=matrix[j][0].multiply(m.getElement(0,k));
                                                        for(n=1;n<array.length;n++)
                                                                array[j][k]=array[j][k].add(matrix[j][n].multiply(m.getElement(n,k)));
                                                }
                                        }
                                        return new ComplexSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private ComplexSquareMatrix rawMultiply(final ComplexSquareMatrix m) {
                if(matrix.length==m.matrix.length) {
                        int n,k;
                        final Complex array[][]=new Complex[matrix.length][matrix.length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array.length;k++) {
                                        array[j][k]=matrix[j][0].multiply(m.matrix[0][k]);
                                        for(n=1;n<array.length;n++)
                                                array[j][k]=array[j][k].add(matrix[j][n].multiply(m.matrix[n][k]));
                                }
                        }
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }

// HERMITIAN ADJOINT

        /**
        * Returns the hermitian adjoint of this matrix.
        * @return a complex square matrix
        */
        public ComplexMatrix hermitianAdjoint() {
                final Complex array[][]=new Complex[matrix.length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[0][i]=matrix[i][0].conjugate();
                        for(j=1;j<array.length;j++)
                                array[j][i]=matrix[i][j].conjugate();
                }
                return new ComplexSquareMatrix(array);
        }

// CONJUGATE

        /**
        * Returns the complex conjugate of this matrix.
        * @return a complex square matrix
        */
        public ComplexMatrix conjugate() {
                final Complex array[][]=new Complex[matrix.length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=matrix[i][0].conjugate();
                        for(j=1;j<array.length;j++)
                                array[i][j]=matrix[i][j].conjugate();
                }
                return new ComplexSquareMatrix(array);
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return a complex square matrix
        */
        public Matrix transpose() {
                final Complex array[][]=new Complex[matrix.length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[0][i]=matrix[i][0];
                        for(j=1;j<array.length;j++)
                                array[j][i]=matrix[i][j];
                }
                return new ComplexSquareMatrix(array);
        }

// INVERSE

        /**
        * Returns the inverse of this matrix.
        * @return a complex square matrix
        */
        public ComplexSquareMatrix inverse() {
                int i,j,k;
                Complex tmpL,tmpU;
                final Complex array[][][]=new Complex[2][matrix.length][matrix.length];
                final ComplexSquareMatrix lu[]=this.luDecompose();
                array[0][0][0]=Complex.ONE.divide(lu[0].matrix[0][0]);
                array[1][0][0]=Complex.ONE.divide(lu[1].matrix[0][0]);
                for(i=1;i<matrix.length;i++) {
                        array[0][i][i]=Complex.ONE.divide(lu[0].matrix[i][i]);
                        array[1][i][i]=Complex.ONE.divide(lu[1].matrix[i][i]);
                }
                for(i=0;i<matrix.length-1;i++) {
                        for(j=i+1;j<matrix.length;j++) {
                                tmpL=tmpU=Complex.ZERO;
                                for(k=i;k<j;k++) {
                                        tmpL=tmpL.subtract((lu[0].matrix[j][k]).multiply(array[0][k][i]));
                                        tmpU=tmpU.subtract(array[1][i][k].multiply(lu[1].matrix[k][j]));
                                }
                                array[0][i][j]=Complex.ZERO;
                                array[0][j][i]=tmpL.divide(lu[0].matrix[j][j]);
                                array[1][j][i]=Complex.ZERO;
                                array[1][i][j]=tmpU.divide(lu[1].matrix[j][j]);
                        }
                }
                // matrix multiply array[1] x array[0]
                final Complex inv[][]=new Complex[matrix.length][matrix.length];
                for(i=0;i<matrix.length;i++) {
                        for(j=0;j<matrix.length;j++) {
                                inv[i][j]=array[1][i][0].multiply(array[0][0][j]);
                                for(k=1;k<matrix.length;k++)
                                        inv[i][j]=inv[i][j].add(array[1][i][k].multiply(array[0][k][j]));
                        }
                }
                return new ComplexSquareMatrix(inv);
        }

// LU DECOMPOSITION

        /**
        * Returns the LU decomposition of this matrix.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public ComplexSquareMatrix[] luDecompose() {
                int i,j,k;
                Complex tmp;
                final Complex array[][][]=new Complex[2][matrix.length][matrix.length];
                array[0][0][0]=Complex.ONE;
                for(i=1;i<matrix.length;i++)
                        array[0][i][i]=Complex.ONE;
                for(j=0;j<matrix.length;j++) {
                        for(i=0;i<=j;i++) {
                                tmp=matrix[i][j];
                                for(k=0;k<i;k++)
                                        tmp=tmp.subtract(array[0][i][k].multiply(array[1][k][j]));
                                array[1][j][i]=Complex.ZERO;
                                array[1][i][j]=tmp;
                        }
                        for(i=j+1;i<matrix.length;i++) {
                                tmp=matrix[i][j];
                                for(k=0;k<j;k++)
                                        tmp=tmp.subtract(array[0][i][k].multiply(array[1][k][j]));
                                array[0][j][i]=Complex.ZERO;
                                array[0][i][j]=tmp.divide(array[1][j][j]);
                        }
                }
                final ComplexSquareMatrix lu[]=new ComplexSquareMatrix[2];
                lu[0]=new ComplexSquareMatrix(array[0]);
                lu[1]=new ComplexSquareMatrix(array[1]);
                return lu;
        }

// MAP ELEMENTS

        /**
        * Applies a function on all the matrix elements.
        * @param f a user-defined function
        * @return a complex square matrix
        */
        public ComplexMatrix mapElements(final Mapping f) {
                final Complex array[][]=new Complex[matrix.length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=f.map(matrix[i][0]);
                        for(j=1;j<array.length;j++)
                                array[i][j]=f.map(matrix[i][j]);
                }
                return new ComplexSquareMatrix(array);
        }
}

