package JSci.maths;

import JSci.GlobalSettings;
import JSci.maths.groups.AbelianGroupMember;

/**
* The DoubleMatrix class provides an object for encapsulating double matrix algebra.
* @version 2.1
* @author Mark Hale
*/
public class DoubleMatrix extends Matrix {
        /**
        * Storage format identifier.
        */
        protected final static int ARRAY_2D=1;
        protected final static int storageFormat=ARRAY_2D;
        /**
        * Array containing the elements of the matrix.
        */
        protected double matrix[][];
        /**
        * Constructs a matrix.
        */
        protected DoubleMatrix() {
                super();
        }
        /**
        * Constructs an empty matrix.
        * @param row the number of rows
        * @param col the number of columns
        */
        public DoubleMatrix(final int row,final int col) {
                this();
                matrix=new double[row][col];
        }
        /**
        * Constructs a matrix by wrapping an array.
        * @param array an assigned value
        */
        public DoubleMatrix(final double array[][]) {
                this();
                matrix=array;
        }
        /**
        * Constructs a matrix from an array of vectors (columns).
        * @param array an assigned value
        */
        public DoubleMatrix(final DoubleVector array[]) {
                this(array[0].dimension(),array.length);
                for(int j,i=0;i<matrix.length;i++) {
                        for(j=0;j<matrix[0].length;j++)
                                matrix[i][j]=array[j].getComponent(i);
                }
        }
        /**
        * Finalize.
        * @exception Throwable Any that occur.
        */
        protected void finalize() throws Throwable {
                matrix=null;
                super.finalize();
        }
        /**
        * Compares two double matrices for equality.
        * @param m a double matrix
        */
        public boolean equals(Object m) {
                if(m!=null && (m instanceof DoubleMatrix) &&
                matrix.length==((DoubleMatrix)m).rows() && matrix[0].length==((DoubleMatrix)m).columns()) {
                        final DoubleMatrix dm=(DoubleMatrix)m;
                        for(int j,i=0;i<matrix.length;i++) {
                                for(j=0;j<matrix[0].length;j++) {
                                        if(Math.abs(matrix[i][j]-dm.getElement(i,j))>GlobalSettings.ZERO_TOL)
                                                return false;
                                }
                        }
                        return true;
                } else
                        return false;
        }
        /**
        * Returns a string representing this matrix.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer(matrix.length*matrix[0].length);
                for(int j,i=0;i<matrix.length;i++) {
                        for(j=0;j<matrix[0].length;j++) {
                                buf.append(matrix[i][j]);
                                buf.append(' ');
                        }
                        buf.append('\n');
                }
                return buf.toString();
        }
        /**
        * Returns a hashcode for this matrix.
        */
        public int hashCode() {
                return (int)Math.exp(infNorm());
        }
        /**
        * Converts this matrix to an integer matrix.
        * @return an integer matrix
        */
        public IntegerMatrix toIntegerMatrix() {
                final int ans[][]=new int[matrix.length][matrix[0].length];
                for(int j,i=0;i<matrix.length;i++) {
                        for(j=0;j<matrix[0].length;j++)
                                ans[i][j]=Math.round((float)matrix[i][j]);
                }
                return new IntegerMatrix(ans);
        }
        /**
        * Converts this matrix to a complex matrix.
        * @return a complex matrix
        */
        public ComplexMatrix toComplexMatrix() {
                final Complex ans[][]=new Complex[matrix.length][matrix[0].length];
                for(int j,i=0;i<matrix.length;i++) {
                        for(j=0;j<matrix[0].length;j++)
                                ans[i][j]=new Complex(matrix[i][j],0.0);
                }
                return new ComplexMatrix(ans);
        }
        /**
        * Returns an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public double getElement(final int i, final int j) {
                if(i>=0 && i<matrix.length && j>=0 && j<matrix[0].length)
                        return matrix[i][j];
                else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Sets the value of an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @param x a number
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public void setElement(final int i, final int j, final double x) {
                if(i>=0 && i<matrix.length && j>=0 && j<matrix[0].length)
                        matrix[i][j]=x;
                else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public double infNorm() {
                double result=0.0,tmpResult;
                for(int j,i=0;i<matrix.length;i++) {
                        tmpResult=0.0;
                        for(j=0;j<matrix[0].length;j++)
                                tmpResult+=Math.abs(matrix[i][j]);
                        if(tmpResult>result)
                                result=tmpResult;
                }
                return result;
        }
        /**
        * Returns the Frobenius norm.
        * @author Taber Smith
        */
        public double frobeniusNorm() {
                double result=0.0;
                for(int j,i=0;i<matrix.length;i++)
                        for(j=0;j<matrix[0].length;j++)
                                result+=matrix[i][j]*matrix[i][j];
                return Math.sqrt(result);
        }
        /**
        * Returns the number of rows.
        */
        public int rows() {
                return matrix.length;
        }
        /**
        * Returns the number of columns.
        */
        public int columns() {
                return matrix[0].length;
        }

//============
// OPERATIONS
//============

        /**
        * Returns the negative of this matrix.
        */
        public AbelianGroupMember negate() {
                final double array[][]=new double[matrix.length][matrix[0].length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=-matrix[i][0];
                        for(j=1;j<array[0].length;j++)
                                array[i][j]=-matrix[i][j];
                }
                return new DoubleMatrix(array);
        }

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        */
        public AbelianGroupMember add(final AbelianGroupMember m) {
                if(m instanceof DoubleMatrix)
                        return add((DoubleMatrix)m);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix add(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        default: 
                                if(matrix.length==m.rows() && matrix[0].length==m.columns()) {
                                        final double array[][]=new double[matrix.length][matrix[0].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0]+m.getElement(i,0);
                                                for(j=1;j<array[0].length;j++)
                                                        array[i][j]=matrix[i][j]+m.getElement(i,j);
                                        }
                                        return new DoubleMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleMatrix rawAdd(final DoubleMatrix m) {
                if(matrix.length==m.matrix.length && matrix[0].length==m.matrix[0].length) {
                        final double array[][]=new double[matrix.length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[i][0]+m.matrix[i][0];
                                for(j=1;j<array[0].length;j++)
                                        array[i][j]=matrix[i][j]+m.matrix[i][j];
                        }
                        return new DoubleMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix by another.
        */
        public AbelianGroupMember subtract(final AbelianGroupMember m) {
                if(m instanceof DoubleMatrix)
                        return subtract((DoubleMatrix)m);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix subtract(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        default: 
                                if(matrix.length==m.rows() && matrix[0].length==m.columns()) {
                                        final double array[][]=new double[matrix.length][matrix[0].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0]-m.getElement(i,0);
                                                for(j=1;j<array[0].length;j++)
                                                        array[i][j]=matrix[i][j]-m.getElement(i,j);
                                        }
                                        return new DoubleMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleMatrix rawSubtract(final DoubleMatrix m) {
                if(matrix.length==m.matrix.length && matrix[0].length==m.matrix[0].length) {
                        final double array[][]=new double[matrix.length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[i][0]-m.matrix[i][0];
                                for(j=1;j<array[0].length;j++)
                                        array[i][j]=matrix[i][j]-m.matrix[i][j];
                        }
                        return new DoubleMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this matrix by a scalar.
        */
        public AlgebraMember scalarMultiply(RingMember x) {
                if(x instanceof MathDouble)
                        return scalarMultiply(((MathDouble)x).value());
                else if(x instanceof MathInteger)
                        return scalarMultiply(((MathInteger)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x a double
        * @return a double matrix
        */
        public DoubleMatrix scalarMultiply(final double x) {
                final double array[][]=new double[matrix.length][matrix[0].length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=x*matrix[i][0];
                        for(j=1;j<array[0].length;j++)
                                array[i][j]=x*matrix[i][j];
                }
                return new DoubleMatrix(array);
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v a double vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public DoubleVector multiply(final DoubleVector v) {
                if(matrix[0].length==v.dimension()) {
                        final double array[]=new double[matrix.length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i]=matrix[i][0]*v.getComponent(0);
                                for(j=1;j<matrix[0].length;j++)
                                        array[i]+=matrix[i][j]*v.getComponent(j);
                        }
                        return new DoubleVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        */
        public RingMember multiply(final RingMember m) {
                if(m instanceof DoubleMatrix)
                        return multiply((DoubleMatrix)m);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double matrix
        * @return a DoubleMatrix or a DoubleSquareMatrix as appropriate
        * @exception MatrixDimensionException If the matrices are incompatible.
        */
        public DoubleMatrix multiply(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        default: 
                                if(matrix[0].length==m.rows()) {
                                        int n,k;
                                        final double array[][]=new double[matrix.length][m.columns()];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array[0].length;k++) {
                                                        array[j][k]=matrix[j][0]*m.getElement(0,k);
                                                        for(n=1;n<matrix[0].length;n++)
                                                                array[j][k]+=matrix[j][n]*m.getElement(n,k);
                                                }
                                        }
                                        if(array.length==array[0].length)
                                                return new DoubleSquareMatrix(array);
                                        else
                                                return new DoubleMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private DoubleMatrix rawMultiply(final DoubleMatrix m) {
                if(matrix[0].length==m.matrix.length) {
                        int n,k;
                        final double array[][]=new double[matrix.length][m.matrix[0].length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array[0].length;k++) {
                                        array[j][k]=matrix[j][0]*m.matrix[0][k];
                                        for(n=1;n<matrix[0].length;n++)
                                                array[j][k]+=matrix[j][n]*m.matrix[n][k];
                                }
                        }
                        if(array.length==array[0].length)
                                return new DoubleSquareMatrix(array);
                        else
                                return new DoubleMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return a double matrix
        */
        public Matrix transpose() {
                final double array[][]=new double[matrix[0].length][matrix.length];
                for(int j,i=0;i<array[0].length;i++) {
                        array[0][i]=matrix[i][0];
                        for(j=1;j<array.length;j++)
                                array[j][i]=matrix[i][j];
                }
                return new DoubleMatrix(array);
        }

// MAP ELEMENTS

        /**
        * Applies a function on all the matrix elements.
        * @param f a user-defined function
        * @return a double matrix
        */
        public DoubleMatrix mapElements(final Mapping f) {
                final double array[][]=new double[matrix.length][matrix[0].length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=f.map(matrix[i][0]);
                        for(j=1;j<array[0].length;j++)
                                array[i][j]=f.map(matrix[i][j]);
                }
                return new DoubleMatrix(array);
        }
}

