package JSci.maths;

/**
* The DoubleSquareMatrix class provides an object for encapsulating square matrices containing doubles.
* @version 2.1
* @author Mark Hale
*/
public class DoubleSquareMatrix extends DoubleMatrix {
        /**
        * Constructs a matrix.
        */
        protected DoubleSquareMatrix() {
                super();
        }
        /**
        * Constructs an empty matrix.
        * @param size the number of rows/columns
        */
        public DoubleSquareMatrix(final int size) {
                super(size,size);
        }
        /**
        * Constructs a matrix by wrapping an array.
        * @param array an assigned value
        * @exception MatrixDimensionException If the array is not square.
        */
        public DoubleSquareMatrix(final double array[][]) {
                super(array);
                if(array.length!=array[0].length) {
                        matrix=null;
                        throw new MatrixDimensionException("The array is not square.");
                }
        }
        /**
        * Constructs a matrix from an array of vectors (columns).
        * @param array an assigned value
        */
        public DoubleSquareMatrix(final DoubleVector array[]) {
                super(array);
                if(array.length!=array[0].dimension()) {
                        matrix=null;
                        throw new MatrixDimensionException("The array does not form a square matrix.");
                }
        }
        /**
        * Returns true if this matrix is symmetric.
        */
        public boolean isSymmetric() {
                return this.equals(this.transpose());
        }
        /**
        * Returns true if this matrix is unitary.
        */
        public boolean isUnitary() {
                return this.multiply(this.transpose()).equals(DoubleDiagonalMatrix.identity(matrix[0].length));
        }
        /**
        * Returns the determinant.
        */
        public double det() {
                if(matrix.length==2) {
                        return matrix[0][0]*matrix[1][1]-matrix[0][1]*matrix[1][0];
                } else {
                        final DoubleSquareMatrix lu[]=this.luDecompose();
                        double det=lu[1].matrix[0][0];
                        for(int i=1;i<matrix.length;i++)
                                det*=lu[1].matrix[i][i];
                        return det;
                }
        }
        /**
        * Returns the trace.
        * @author Taber Smith
        */
        public double trace() {
                double result=matrix[0][0];
                for(int i=1;i<matrix.length;i++)
                        result+=matrix[i][i];
                return result;
        }

//============
// OPERATIONS
//============

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix add(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        default: 
                                if(matrix.length==m.rows() && matrix.length==m.columns()) {
                                        final double array[][]=new double[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0]+m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j]+m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleSquareMatrix rawAdd(final DoubleMatrix m) {
                if(matrix.length==m.matrix.length && matrix.length==m.matrix[0].length) {
                        final double array[][]=new double[matrix.length][matrix.length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[i][0]+m.matrix[i][0];
                                for(j=1;j<array.length;j++)
                                        array[i][j]=matrix[i][j]+m.matrix[i][j];
                        }
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix add(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        default: 
                                if(matrix.length==m.rows()) {
                                        final double array[][]=new double[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0]+m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j]+m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix subtract(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        default: 
                                if(matrix.length==m.rows() && matrix.length==m.columns()) {
                                        final double array[][]=new double[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0]-m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j]-m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleSquareMatrix rawSubtract(final DoubleMatrix m) {
                if(matrix.length==m.matrix.length && matrix.length==m.matrix[0].length) {
                        final double array[][]=new double[matrix.length][matrix.length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[i][0]-m.matrix[i][0];
                                for(j=1;j<array.length;j++)
                                        array[i][j]=matrix[i][j]-m.matrix[i][j];
                        }
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix subtract(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        default: 
                                if(matrix.length==m.rows()) {
                                        final double array[][]=new double[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0]-m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j]-m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x a double
        * @return a double square matrix
        */
        public DoubleMatrix scalarMultiply(final double x) {
                final double array[][]=new double[matrix.length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=x*matrix[i][0];
                        for(j=1;j<array.length;j++)
                                array[i][j]=x*matrix[i][j];
                }
                return new DoubleSquareMatrix(array);
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v a double vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public DoubleVector multiply(final DoubleVector v) {
                if(matrix.length==v.dimension()) {
                        final double array[]=new double[matrix.length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i]=matrix[i][0]*v.getComponent(0);
                                for(j=1;j<matrix.length;j++)
                                        array[i]+=matrix[i][j]*v.getComponent(j);
                        }
                        return new DoubleVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are incompatible.
        */
        public DoubleMatrix multiply(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        default:
                                if(matrix[0].length==m.rows()) {
                                        int n,k;
                                        final double array[][]=new double[matrix.length][m.columns()];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array[0].length;k++) {
                                                        array[j][k]=matrix[j][0]*m.getElement(0,k);
                                                        for(n=1;n<matrix[0].length;n++)
                                                                array[j][k]+=matrix[j][n]*m.getElement(n,k);
                                                }
                                        }
                                        return new DoubleMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private DoubleMatrix rawMultiply(final DoubleMatrix m) {
                if(matrix[0].length==m.matrix.length) {
                        int n,k;
                        final double array[][]=new double[matrix.length][m.matrix[0].length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array[0].length;k++) {
                                        array[j][k]=matrix[j][0]*m.matrix[0][k];
                                        for(n=1;n<matrix[0].length;n++)
                                                array[j][k]+=matrix[j][n]*m.matrix[n][k];
                                }
                        }
                        return new DoubleMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are incompatible.
        */
        public DoubleSquareMatrix multiply(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        default:
                                if(matrix.length==m.rows()) {
                                        int n,k;
                                        final double array[][]=new double[matrix.length][matrix.length];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array.length;k++) {
                                                        array[j][k]=matrix[j][0]*m.getElement(0,k);
                                                        for(n=1;n<array.length;n++)
                                                                array[j][k]+=matrix[j][n]*m.getElement(n,k);
                                                }
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private DoubleSquareMatrix rawMultiply(final DoubleSquareMatrix m) {
                if(matrix.length==m.matrix.length) {
                        int n,k;
                        final double array[][]=new double[matrix.length][matrix.length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array.length;k++) {
                                        array[j][k]=matrix[j][0]*m.matrix[0][k];
                                        for(n=1;n<array.length;n++)
                                                array[j][k]+=matrix[j][n]*m.matrix[n][k];
                                }
                        }
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return a double square matrix
        */
        public Matrix transpose() {
                final double array[][]=new double[matrix.length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[0][i]=matrix[i][0];
                        for(j=1;j<array.length;j++)
                                array[j][i]=matrix[i][j];
                }
                return new DoubleSquareMatrix(array);
        }

// INVERSE

        /**
        * Returns the inverse of this matrix.
        * @return a double square matrix
        */
        public DoubleSquareMatrix inverse() {
                int i,j,k;
                double tmpL,tmpU;
                final double array[][][]=new double[2][matrix.length][matrix.length];
                final DoubleSquareMatrix lu[]=this.luDecompose();
                array[0][0][0]=1.0/lu[0].matrix[0][0];
                array[1][0][0]=1.0/lu[1].matrix[0][0];
                for(i=1;i<matrix.length;i++) {
                        array[0][i][i]=1.0/lu[0].matrix[i][i];
                        array[1][i][i]=1.0/lu[1].matrix[i][i];
                }
                for(i=0;i<matrix.length-1;i++) {
                        for(j=i+1;j<matrix.length;j++) {
                                tmpL=tmpU=0.0;
                                for(k=i;k<j;k++) {
                                        tmpL-=lu[0].matrix[j][k]*array[0][k][i];
                                        tmpU-=array[1][i][k]*lu[1].matrix[k][j];
                                }
                                array[0][j][i]=tmpL/lu[0].matrix[j][j];
                                array[1][i][j]=tmpU/lu[1].matrix[j][j];
                        }
                }
                // matrix multiply array[1] x array[0]
                final double inv[][]=new double[matrix.length][matrix.length];
                for(i=0;i<matrix.length;i++) {
                        for(j=0;j<matrix.length;j++) {
                                inv[i][j]=array[1][i][0]*array[0][0][j];
                                for(k=1;k<matrix.length;k++)
                                        inv[i][j]+=array[1][i][k]*array[0][k][j];
                        }
                }
                return new DoubleSquareMatrix(inv);
        }

// LU DECOMPOSITION

        /**
        * Returns the LU decomposition of this matrix.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] luDecompose() {
                int i,j,k;
                double tmp;
                final double array[][][]=new double[2][matrix.length][matrix.length];
                array[0][0][0]=1.0;
                for(i=1;i<matrix.length;i++)
                        array[0][i][i]=1.0;
                for(j=0;j<matrix.length;j++) {
                        for(i=0;i<=j;i++) {
                                tmp=matrix[i][j];
                                for(k=0;k<i;k++)
                                        tmp-=array[0][i][k]*array[1][k][j];
                                array[1][i][j]=tmp;
                        }
                        for(i=j+1;i<matrix.length;i++) {
                                tmp=matrix[i][j];
                                for(k=0;k<j;k++)
                                        tmp-=array[0][i][k]*array[1][k][j];
                                array[0][i][j]=tmp/array[1][j][j];
                        }
                }
                final DoubleSquareMatrix lu[]=new DoubleSquareMatrix[2];
                lu[0]=new DoubleSquareMatrix(array[0]);
                lu[1]=new DoubleSquareMatrix(array[1]);
                return lu;
        }

// CHOLESKY DECOMPOSITION

        /**
        * Returns the Cholesky decomposition of this matrix.
        * Matrix must be symmetric and positive definite.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] choleskyDecompose() {
                int i,j,k;
                double tmp;
                final double array[][][]=new double[2][matrix.length][matrix.length];
                array[0][0][0]=array[1][0][0]=Math.sqrt(matrix[0][0]);
                for(i=1;i<matrix.length;i++)
                        array[0][i][0]=array[1][0][i]=matrix[i][0]/array[0][0][0];
                for(j=1;j<matrix.length;j++) {
                        tmp=matrix[j][j];
                        for(i=0;i<j;i++)
                                tmp-=array[0][j][i]*array[0][j][i];
                        array[0][j][j]=array[1][j][j]=Math.sqrt(tmp);
                        for(i=j+1;i<matrix.length;i++) {
                                tmp=matrix[i][j];
                                for(k=0;k<i;k++)
                                        tmp-=array[0][j][k]*array[1][k][i];
                                array[0][i][j]=array[1][j][i]=tmp/array[1][j][j];
                        }
                }
                final DoubleSquareMatrix lu[]=new DoubleSquareMatrix[2];
                lu[0]=new DoubleSquareMatrix(array[0]);
                lu[1]=new DoubleSquareMatrix(array[1]);
                return lu;
        }

// MAP ELEMENTS

        /**
        * Applies a function on all the matrix elements.
        * @param f a user-defined function
        * @return a double square matrix
        */
        public DoubleMatrix mapElements(final Mapping f) {
                final double array[][]=new double[matrix.length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=f.map(matrix[i][0]);
                        for(j=1;j<array.length;j++)
                                array[i][j]=f.map(matrix[i][j]);
                }
                return new DoubleSquareMatrix(array);
        }
}

