package JSci.maths;

import JSci.GlobalSettings;
import JSci.maths.groups.AbelianGroupMember;

/**
* The DoubleVector class encapsulates double vectors.
* @version 2.1
* @author Mark Hale
*/
public class DoubleVector extends MathVector {
        /**
        * Storage format identifier.
        */
        protected final static int ARRAY_1D=1;
        protected final static int storageFormat=ARRAY_1D;
        /**
        * Array containing the components of the vector.
        */
        protected double vector[];
        /**
        * Constructs an empty vector.
        * @param dim the dimension of the vector.
        */
        public DoubleVector(final int dim) {
                vector=new double[dim];
        }
        /**
        * Constructs a vector by wrapping an array.
        * @param array an assigned value
        */
        public DoubleVector(final double array[]) {
                vector=array;
        }
        /**
        * Compares two double vectors for equality.
        * @param a a double vector
        */
        public boolean equals(Object a) {
                if(a!=null && (a instanceof DoubleVector) && vector.length==((DoubleVector)a).dimension()) {
                        final DoubleVector dv=(DoubleVector)a;
                        for(int i=0;i<vector.length;i++) {
                                if(Math.abs(vector[i]-dv.getComponent(i))>GlobalSettings.ZERO_TOL)
                                        return false;
                        }
                        return true;
                } else
                        return false;
        }
        /**
        * Returns a comma delimited string representing the value of this vector.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer(vector.length);
                int i;
                for(i=0;i<vector.length-1;i++) {
                        buf.append(vector[i]);
                        buf.append(',');
                }
                buf.append(vector[i]);
                return buf.toString();
        }
        /**
        * Returns a hashcode for this vector.
        */
        public int hashCode() {
                return (int)Math.exp(norm());
        }
        /**
        * Converts this vector to an integer vector.
        * @return an integer vector
        */
        public IntegerVector toIntegerVector() {
                final int ans[]=new int[vector.length];
                for(int i=0;i<vector.length;i++)
                        ans[i]=Math.round((float)vector[i]);
                return new IntegerVector(ans);
        }
        /**
        * Converts this vector to a complex vector.
        * @return a complex vector
        */
        public ComplexVector toComplexVector() {
                final Complex ans[]=new Complex[vector.length];
                for(int i=0;i<vector.length;i++)
                        ans[i]=new Complex(vector[i],0.0);
                return new ComplexVector(ans);
        }
        /**
        * Returns a component of this vector.
        * @param n index of the vector component
        * @exception VectorDimensionException If attempting to access an invalid component.
        */
        public double getComponent(final int n) {
                if(n>=0 && n<vector.length)
                        return vector[n];
                else
                        throw new VectorDimensionException("Invalid component.");
        }
        /**
        * Sets the value of a component of this vector.
        * @param n index of the vector component
        * @param x a number
        * @exception VectorDimensionException If attempting to access an invalid component.
        */
        public void setComponent(final int n, final double x) {
                if(n>=0 && n<vector.length)
                        vector[n]=x;
                else
                        throw new VectorDimensionException("Invalid component.");
        }
        /**
        * Returns the vector's dimension.
        */
        public int dimension() {
                return vector.length;
        }
        /**
        * Returns the ln-norm.
        */
        public double norm(int n) {
                double answer=Math.pow(vector[0],n);
                for(int i=1;i<vector.length;i++)
                        answer+=Math.pow(vector[i],n);
                return Math.pow(answer,1.0/n);
        }
        /**
        * Returns the l2-norm (magnitude).
        */
        public double norm() {
                double answer=vector[0]*vector[0];
                for(int i=1;i<vector.length;i++)
                        answer+=vector[i]*vector[i];
                return Math.sqrt(answer);
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public double infNorm() {
                double infNorm=vector[0];
                for(int i=1;i<vector.length;i++) {
                        if(vector[i]>infNorm)
                                infNorm=vector[i];
                }
                return infNorm;
        }

//============
// OPERATIONS
//============

        /**
        * Returns the negative of this vector.
        */
        public AbelianGroupMember negate() {
                final double array[]=new double[vector.length];
                array[0]=-vector[0];
                for(int i=1;i<array.length;i++)
                        array[i]=-vector[i];
                return new DoubleVector(array);
        }

// ADDITION

        /**
        * Returns the addition of this vector and another.
        */
        public AbelianGroupMember add(final AbelianGroupMember v) {
                if(v instanceof DoubleVector)
                        return add((DoubleVector)v);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the addition of this vector and another.
        * @param v a double vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public DoubleVector add(final DoubleVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawAdd(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        final double array[]=new double[vector.length];
                                        array[0]=vector[0]+v.getComponent(0);
                                        for(int i=1;i<array.length;i++)
                                                array[i]=vector[i]+v.getComponent(i);
                                        return new DoubleVector(array);
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private DoubleVector rawAdd(final DoubleVector v) {
                if(vector.length==v.vector.length) {
                        final double array[]=new double[vector.length];
                        array[0]=vector[0]+v.vector[0];
                        for(int i=1;i<array.length;i++)
                                array[i]=vector[i]+v.vector[i];
                        return new DoubleVector(array);
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this vector by another.
        */
        public AbelianGroupMember subtract(final AbelianGroupMember v) {
                if(v instanceof DoubleVector)
                        return subtract((DoubleVector)v);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the subtraction of this vector by another.
        * @param v a double vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public DoubleVector subtract(final DoubleVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawSubtract(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        final double array[]=new double[vector.length];
                                        array[0]=vector[0]-v.getComponent(0);
                                        for(int i=1;i<array.length;i++)
                                                array[i]=vector[i]-v.getComponent(i);
                                        return new DoubleVector(array);
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private DoubleVector rawSubtract(final DoubleVector v) {
                if(vector.length==v.vector.length) {
                        final double array[]=new double[vector.length];
                        array[0]=vector[0]-v.vector[0];
                        for(int i=1;i<array.length;i++)
                                array[i]=vector[i]-v.vector[i];
                        return new DoubleVector(array);
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this vector by a scalar.
        */
        public ModuleMember scalarMultiply(RingMember x) {
                if(x instanceof MathDouble)
                        return scalarMultiply(((MathDouble)x).value());
                else if(x instanceof MathInteger)
                        return scalarMultiply(((MathInteger)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the multiplication of this vector by a scalar.
        * @param x a double
        */
        public DoubleVector scalarMultiply(final double x) {
                final double array[]=new double[vector.length];
                array[0]=x*vector[0];
                for(int i=1;i<array.length;i++)
                        array[i]=x*vector[i];
                return new DoubleVector(array);
        }

// SCALAR DIVISION

        /**
        * Returns the division of this vector by a scalar.
        * @param x a double
        * @exception ArithmeticException If divide by zero.
        */
        public DoubleVector scalarDivide(final double x) {
                final double array[]=new double[vector.length];
                array[0]=vector[0]/x;
                for(int i=1;i<array.length;i++)
                        array[i]=vector[i]/x;
                return new DoubleVector(array);
        }

// SCALAR PRODUCT

        /**
        * Returns the scalar product of this vector and another.
        * @param v a double vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public double scalarProduct(final DoubleVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawScalarProduct(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        double answer=vector[0]*v.getComponent(0);
                                        for(int i=1;i<vector.length;i++)
                                                answer+=vector[i]*v.getComponent(i);
                                        return answer;
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private double rawScalarProduct(final DoubleVector v) {
                if(vector.length==v.vector.length) {
                        double answer=vector[0]*v.vector[0];
                        for(int i=1;i<vector.length;i++)
                                answer+=vector[i]*v.vector[i];
                        return answer;
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }

// MAP COMPONENTS

        /**
        * Applies a function on all the vector components.
        * @param f a user-defined function
        * @return a double vector
        */
        public DoubleVector mapComponents(final Mapping f) {
                final double array[]=new double[vector.length];
                array[0]=f.map(vector[0]);
                for(int i=1;i<vector.length;i++)
                        array[i]=f.map(vector[i]);
                return new DoubleVector(array);
        }
}

