package JSci.maths;

/**
* The IntegerSquareMatrix class provides an object for encapsulating square matrices containing integers.
* @version 2.1
* @author Mark Hale
*/
public class IntegerSquareMatrix extends IntegerMatrix {
        /**
        * Constructs a matrix.
        */
        protected IntegerSquareMatrix() {
                super();
        }
        /**
        * Constructs an empty matrix.
        * @param size the number of rows/columns
        */
        public IntegerSquareMatrix(final int size) {
                super(size,size);
        }
        /**
        * Constructs a matrix by wrapping an array.
        * @param array an assigned value
        * @exception MatrixDimensionException If the array is not square.
        */
        public IntegerSquareMatrix(final int array[][]) {
                super(array);
                if(array.length!=array[0].length) {
                        matrix=null;
                        throw new MatrixDimensionException("The array is not square.");
                }
        }
        /**
        * Constructs a matrix from an array of vectors (columns).
        * @param array an assigned value
        */
        public IntegerSquareMatrix(final IntegerVector array[]) {
                super(array);
                if(array.length!=array[0].dimension()) {
                        matrix=null;
                        throw new MatrixDimensionException("The array does not form a square matrix.");
                }
        }
        /**
        * Returns true if this matrix is symmetric.
        */
        public boolean isSymmetric() {
                return this.equals(this.transpose());
        }
        /**
        * Returns true if this matrix is unitary.
        */
        public boolean isUnitary() {
                return this.multiply(this.transpose()).equals(IntegerDiagonalMatrix.identity(matrix[0].length));
        }
        /**
        * Returns the determinant.
        */
        public int det() {
                if(matrix.length==2) {
                        return matrix[0][0]*matrix[1][1]-matrix[0][1]*matrix[1][0];
                } else {
                        final DoubleSquareMatrix lu[]=this.luDecompose();
                        double det=lu[1].matrix[0][0];
                        for(int i=1;i<matrix.length;i++)
                                det*=lu[1].matrix[i][i];
                        return (int)(det);
                }
        }
        /**
        * Returns the trace.
        * @author Taber Smith
        */
        public int trace() {
                int result=matrix[0][0];
                for (int i=1;i<matrix.length;i++)
                        result+=matrix[i][i];
                return result;
        }

//============
// OPERATIONS
//============

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        * @param m an integer matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerMatrix add(final IntegerMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        default: 
                                if(matrix.length==m.rows() && matrix.length==m.columns()) {
                                        final int array[][]=new int[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0]+m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j]+m.getElement(i,j);
                                        }
                                        return new IntegerSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private IntegerSquareMatrix rawAdd(final IntegerMatrix m) {
                if(matrix.length==m.matrix.length && matrix.length==m.matrix[0].length) {
                        final int array[][]=new int[matrix.length][matrix.length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[i][0]+m.matrix[i][0];
                                for(j=1;j<array.length;j++)
                                        array[i][j]=matrix[i][j]+m.matrix[i][j];
                        }
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m an integer square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerSquareMatrix add(final IntegerSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        default: 
                                if(matrix.length==m.rows()) {
                                        final int array[][]=new int[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0]+m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j]+m.getElement(i,j);
                                        }
                                        return new IntegerSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix by another.
        * @param m an integer matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerMatrix subtract(final IntegerMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        default: 
                                if(matrix.length==m.rows() && matrix.length==m.columns()) {
                                        final int array[][]=new int[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0]-m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j]-m.getElement(i,j);
                                        }
                                        return new IntegerSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private IntegerSquareMatrix rawSubtract(final IntegerMatrix m) {
                if(matrix.length==m.matrix.length && matrix.length==m.matrix[0].length) {
                        final int array[][]=new int[matrix.length][matrix.length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[i][0]-m.matrix[i][0];
                                for(j=1;j<array.length;j++)
                                        array[i][j]=matrix[i][j]-m.matrix[i][j];
                        }
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m an integer square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerSquareMatrix subtract(final IntegerSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        default: 
                                if(matrix.length==m.rows()) {
                                        final int array[][]=new int[matrix.length][matrix.length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[i][0]-m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[i][j]-m.getElement(i,j);
                                        }
                                        return new IntegerSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x an integer
        * @return an integer square matrix
        */
        public IntegerMatrix scalarMultiply(final int x) {
                final int array[][]=new int[matrix.length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[i][0]=x*matrix[i][0];
                        for(j=1;j<array.length;j++)
                                array[i][j]=x*matrix[i][j];
                }
                return new IntegerSquareMatrix(array);
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v an integer vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public IntegerVector multiply(final IntegerVector v) {
                if(matrix.length==v.dimension()) {
                        final int array[]=new int[matrix.length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i]=matrix[i][0]*v.getComponent(0);
                                for(j=1;j<matrix.length;j++)
                                        array[i]+=matrix[i][j]*v.getComponent(j);
                        }
                        return new IntegerVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m an integer matrix
        * @exception MatrixDimensionException If the matrices are incompatible.
        */
        public IntegerMatrix multiply(final IntegerMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        default:
                                if(matrix[0].length==m.rows()) {
                                        int n,k;
                                        final int array[][]=new int[matrix.length][m.columns()];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array[0].length;k++) {
                                                        array[j][k]=matrix[j][0]*m.getElement(0,k);
                                                        for(n=1;n<matrix[0].length;n++)
                                                                array[j][k]+=matrix[j][n]*m.getElement(n,k);
                                                }
                                        }
                                        return new IntegerMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private IntegerMatrix rawMultiply(final IntegerMatrix m) {
                if(matrix[0].length==m.matrix.length) {
                        int n,k;
                        final int array[][]=new int[matrix.length][m.matrix[0].length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array[0].length;k++) {
                                        array[j][k]=matrix[j][0]*m.matrix[0][k];
                                        for(n=1;n<matrix[0].length;n++)
                                                array[j][k]+=matrix[j][n]*m.matrix[n][k];
                                }
                        }
                        return new IntegerMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m an integer square matrix
        * @exception MatrixDimensionException If the matrices are incompatible.
        */
        public IntegerSquareMatrix multiply(final IntegerSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        default:
                                if(matrix.length==m.rows()) {
                                        int n,k;
                                        final int array[][]=new int[matrix.length][matrix.length];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array.length;k++) {
                                                        array[j][k]=matrix[j][0]*m.getElement(0,k);
                                                        for(n=1;n<array.length;n++)
                                                                array[j][k]+=matrix[j][n]*m.getElement(n,k);
                                                }
                                        }
                                        return new IntegerSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private IntegerSquareMatrix rawMultiply(final IntegerSquareMatrix m) {
                if(matrix.length==m.matrix.length) {
                        int n,k;
                        final int array[][]=new int[matrix.length][matrix.length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array.length;k++) {
                                        array[j][k]=matrix[j][0]*m.matrix[0][k];
                                        for(n=1;n<array.length;n++)
                                                array[j][k]+=matrix[j][n]*m.matrix[n][k];
                                }
                        }
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return an integer square matrix
        */
        public Matrix transpose() {
                final int array[][]=new int[matrix.length][matrix.length];
                for(int j,i=0;i<array.length;i++) {
                        array[0][i]=matrix[i][0];
                        for(j=1;j<array.length;j++)
                                array[j][i]=matrix[i][j];
                }
                return new IntegerSquareMatrix(array);
        }

// LU DECOMPOSITION

        /**
        * Returns the LU decomposition of this matrix.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] luDecompose() {
                int i,j,k;
                double tmp;
                final double array[][][]=new double[2][matrix.length][matrix.length];
                array[0][0][0]=1.0;
                for(i=1;i<matrix.length;i++)
                        array[0][i][i]=1.0;
                for(j=0;j<matrix.length;j++) {
                        for(i=0;i<=j;i++) {
                                tmp=matrix[i][j];
                                for(k=0;k<i;k++)
                                        tmp-=array[0][i][k]*array[1][k][j];
                                array[1][i][j]=tmp;
                        }
                        for(i=j+1;i<matrix.length;i++) {
                                tmp=matrix[i][j];
                                for(k=0;k<j;k++)
                                        tmp-=array[0][i][k]*array[1][k][j];
                                array[0][i][j]=tmp/array[1][j][j];
                        }
                }
                final DoubleSquareMatrix lu[]=new DoubleSquareMatrix[2];
                lu[0]=new DoubleSquareMatrix(array[0]);
                lu[1]=new DoubleSquareMatrix(array[1]);
                return lu;
        }

// CHOLESKY DECOMPOSITION

        /**
        * Returns the Cholesky decomposition of this matrix.
        * Matrix must be symmetric and positive definite.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] choleskyDecomposition() {
                int i,j,k;
                double tmp;
                final double array[][][]=new double[2][matrix.length][matrix.length];
                array[0][0][0]=array[1][0][0]=Math.sqrt(matrix[0][0]);
                for(i=1;i<matrix.length;i++)
                        array[0][i][0]=array[1][0][i]=matrix[i][0]/array[0][0][0];
                for(j=1;j<matrix.length;j++) {
                        tmp=matrix[j][j];
                        for(i=0;i<j;i++)
                                tmp-=array[0][j][i]*array[0][j][i];
                        array[0][j][j]=array[1][j][j]=Math.sqrt(tmp);
                        for(i=j+1;i<matrix.length;i++) {
                                tmp=matrix[i][j];
                                for(k=0;k<i;k++)
                                        tmp-=array[0][j][k]*array[1][k][i];
                                array[0][i][j]=array[1][j][i]=tmp/array[1][j][j];
                        }
                }
                final DoubleSquareMatrix lu[]=new DoubleSquareMatrix[2];
                lu[0]=new DoubleSquareMatrix(array[0]);
                lu[1]=new DoubleSquareMatrix(array[1]);
                return lu;
        }
}

