package JSci.maths;

/**
* The IntegerTridiagonalMatrix class provides an object for encapsulating tridiagonal matrices containing integers.
* Uses compressed diagonal storage.
* @version 2.1
* @author Mark Hale
*/
public class IntegerTridiagonalMatrix extends IntegerSquareMatrix {
//
// example: element storage for a 3x3 tridiagonal matrix
//
// matrix[1][0] matrix[2][0]
// matrix[0][1] matrix[1][1] matrix[2][1]
//              matrix[0][2] matrix[1][2]
//
        /**
        * Storage format identifier.
        */
        protected final static int TRIDIAGONAL=3;
        protected final static int storageFormat=TRIDIAGONAL;
        /**
        * Constructs a matrix.
        */
        protected IntegerTridiagonalMatrix() {
                super();
        }
        /**
        * Constructs an empty matrix.
        * @param size the number of rows/columns
        */
        public IntegerTridiagonalMatrix(final int size) {
                this();
                matrix=new int[3][size];
        }
        /**
        * Constructs a matrix from an array.
        * Any non-tridiagonal elements in the array are ignored.
        * @param array an assigned value
        * @exception MatrixDimensionException If the array is not square.
        */
        public IntegerTridiagonalMatrix(final int array[][]) {
                this(array.length);
                if(array.length==array[0].length) {
                        matrix[1][0]=array[0][0];
                        matrix[2][0]=array[0][1];
                        int i=1;
                        for(;i<array.length-1;i++) {
                                matrix[0][i]=array[i][i-1];
                                matrix[1][i]=array[i][i];
                                matrix[2][i]=array[i][i+1];
                        }
                        matrix[0][i]=array[i][i-1];
                        matrix[1][i]=array[i][i];
                } else {
                        matrix=null;
                        throw new MatrixDimensionException("The array is not square.");
                }
        }
        /**
        * Compares two integer tridiagonal matrices for equality.
        * @param m a integer tridiagonal matrix
        */
        public boolean equals(Object m) {
                if(m!=null && (m instanceof IntegerTridiagonalMatrix) &&
                matrix[1].length==((IntegerTridiagonalMatrix)m).rows()) {
                        final IntegerTridiagonalMatrix itm=(IntegerTridiagonalMatrix)m;
                        if(matrix[1][0]!=itm.getElement(0,0))
                                return false;
                        if(matrix[2][0]!=itm.getElement(0,1))
                                return false;
                        int i=1;
                        for(;i<matrix[1].length-1;i++) {
                                if(matrix[0][i]!=itm.getElement(i,i-1))
                                        return false;
                                if(matrix[1][i]!=itm.getElement(i,i))
                                        return false;
                                if(matrix[2][i]!=itm.getElement(i,i+1))
                                        return false;
                        }
                        if(matrix[0][i]!=itm.getElement(i,i-1))
                                return false;
                        if(matrix[1][i]!=itm.getElement(i,i))
                                return false;
                        return true;
                } else {
                        return false;
                }
        }
        /**
        * Returns a string representing this matrix.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer(matrix.length*matrix[1].length);
                for(int j,i=0;i<rows();i++) {
                        for(j=0;j<columns();j++) {
                                buf.append(getElement(i,j));
                                buf.append(' ');
                        }
                        buf.append('\n');
                }
                return buf.toString();
        }
        /**
        * Converts this matrix to a double matrix.
        * @return a double tridiagonal matrix
        */
        public DoubleMatrix toDoubleMatrix() {
                final DoubleTridiagonalMatrix m=new DoubleTridiagonalMatrix(matrix[1].length);
                m.matrix[1][0]=matrix[1][0];
                m.matrix[2][0]=matrix[2][0];
                int i=1;
                for(;i<matrix[1].length-1;i++) {
                        m.matrix[0][i]=matrix[0][i];
                        m.matrix[1][i]=matrix[1][i];
                        m.matrix[2][i]=matrix[2][i];
                }
                m.matrix[0][i]=matrix[0][i];
                m.matrix[1][i]=matrix[1][i];
                return m;
        }
        /**
        * Converts this matrix to a complex matrix.
        * @return a complex tridiagonal matrix
        */
        public ComplexMatrix toComplexMatrix() {
                final ComplexTridiagonalMatrix m=new ComplexTridiagonalMatrix(matrix[1].length);
                m.matrix[1][0]=new Complex(matrix[1][0],0.0);
                m.matrix[2][0]=new Complex(matrix[2][0],0.0);
                int i=1;
                for(;i<matrix[1].length-1;i++) {
                        m.matrix[0][i]=new Complex(matrix[0][i],0.0);
                        m.matrix[1][i]=new Complex(matrix[1][i],0.0);
                        m.matrix[2][i]=new Complex(matrix[2][i],0.0);
                }
                m.matrix[0][i]=new Complex(matrix[0][i],0.0);
                m.matrix[1][i]=new Complex(matrix[1][i],0.0);
                return m;
        }
        /**
        * Returns an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public int getElement(final int i, final int j) {
                if(i>=0 && i<matrix[1].length && j>=0 && j<matrix[1].length) {
                        if(j==i-1)
                                return matrix[0][i];
                        else if(j==i)
                                return matrix[1][i];
                        else if(j==i+1)
                                return matrix[2][i];
                        else
                                return 0;
                } else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Sets the value of an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @param x a number
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public void setElement(final int i, final int j, final int x) {
                if(i>=0 && i<matrix[1].length && j>=0 && j<matrix[1].length) {
                        if(j==i-1)
                                matrix[0][i]=x;
                        else if(j==i)
                                matrix[1][i]=x;
                        else if(j==i+1)
                                matrix[2][i]=x;
                        else
                                throw new MatrixDimensionException("Invalid element.");
                } else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Returns true if this matrix is symmetric.
        */
        public boolean isSymmetric() {
                if(matrix[0][1]!=matrix[2][0])
                        return false;
                for(int i=1;i<matrix[1].length-1;i++) {
                        if(matrix[0][i+1]!=matrix[2][i])
                                return false;
                }
                return true;
        }
        /**
        * Returns the trace.
        */
        public int trace() {
                int tr=matrix[1][0];
                for(int i=1;i<matrix[1].length;i++)
                        tr+=matrix[1][i];
                return tr;
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public int infNorm() {
                int result=Math.abs(matrix[1][0])+Math.abs(matrix[2][0]);
                int tmpResult;
                int i=1;
                for(;i<matrix[1].length-1;i++) {
                        tmpResult=Math.abs(matrix[0][i])+Math.abs(matrix[1][i])+Math.abs(matrix[2][i]);
                        if(tmpResult>result)
                                result=tmpResult;
                }
                tmpResult=Math.abs(matrix[0][i])+Math.abs(matrix[1][i]);
                if(tmpResult>result)
                        result=tmpResult;
                return result;
        }
        /**
        * Returns the Frobenius norm.
        * @author Taber Smith
        */
        public double frobeniusNorm() {
                int result=matrix[1][0]*matrix[1][0]+matrix[2][0]*matrix[2][0];
                int i=1;
                for(;i<matrix[1].length-1;i++)
                        result+=matrix[0][i]*matrix[0][i]+matrix[1][i]*matrix[1][i]+matrix[2][i]*matrix[2][i];
                result+=matrix[0][i]*matrix[0][i]+matrix[1][i]*matrix[1][i];
                return Math.sqrt(result);
        }
        /**
        * Returns the number of rows.
        */
        public int rows() {
                return matrix[1].length;
        }
        /**
        * Returns the number of columns.
        */
        public int columns() {
                return matrix[1].length;
        }

//============
// OPERATIONS
//============

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        * @param m an integer matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerMatrix add(final IntegerMatrix m) {
                if(m instanceof IntegerTridiagonalMatrix)
                        return add((IntegerTridiagonalMatrix)m);
                else if(m instanceof IntegerSquareMatrix)
                        return rawAdd((IntegerSquareMatrix)m);
                else
                        return rawAdd(m);
        }
        private IntegerMatrix rawAdd(final IntegerMatrix m) {
                if(matrix[1].length==m.rows() && matrix[1].length==m.columns()) {
                        final int array[][]=new int[matrix[1].length][matrix[1].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=getElement(i,0)+m.getElement(i,0);
                                for(j=1;j<array.length;j++)
                                        array[i][j]=getElement(i,j)+m.getElement(i,j);
                        }
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m an integer square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerSquareMatrix add(final IntegerSquareMatrix m) {
                if(m instanceof IntegerTridiagonalMatrix)
                        return add((IntegerTridiagonalMatrix)m);
                else
                        return rawAdd(m);
        }
        private IntegerSquareMatrix rawAdd(final IntegerSquareMatrix m) {
                if(matrix[1].length==m.rows()) {
                        final int array[][]=new int[matrix[1].length][matrix[1].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=getElement(i,0)+m.getElement(i,0);
                                for(j=1;j<array.length;j++)
                                        array[i][j]=getElement(i,j)+m.getElement(i,j);
                        }
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m an integer tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerTridiagonalMatrix add(final IntegerTridiagonalMatrix m) {
                int mRow=matrix[1].length;
                if(mRow==m.rows()) {
                        final IntegerTridiagonalMatrix ans=new IntegerTridiagonalMatrix(mRow);
                        ans.matrix[1][0]=matrix[1][0]+m.getElement(0,0);
                        ans.matrix[2][0]=matrix[2][0]+m.getElement(0,1);
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.matrix[0][i]=matrix[0][i]+m.getElement(i,i-1);
                                ans.matrix[1][i]=matrix[1][i]+m.getElement(i,i);
                                ans.matrix[2][i]=matrix[2][i]+m.getElement(i,i+1);
                        }
                        ans.matrix[0][mRow]=matrix[0][mRow]+m.getElement(mRow,mRow-1);
                        ans.matrix[1][mRow]=matrix[1][mRow]+m.getElement(mRow,mRow);
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix and another.
        * @param m an integer matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerMatrix subtract(final IntegerMatrix m) {
                if(m instanceof IntegerTridiagonalMatrix)
                        return subtract((IntegerTridiagonalMatrix)m);
                else if(m instanceof IntegerSquareMatrix)
                        return rawSubtract((IntegerSquareMatrix)m);
                else
                        return rawSubtract(m);
        }
        private IntegerMatrix rawSubtract(final IntegerMatrix m) {
                if(matrix[1].length==m.rows() && matrix[1].length==m.columns()) {
                        final int array[][]=new int[matrix[1].length][matrix[1].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=getElement(i,0)-m.getElement(i,0);
                                for(j=1;j<array.length;j++)
                                        array[i][j]=getElement(i,j)-m.getElement(i,j);
                        }
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m an integer square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerSquareMatrix subtract(final IntegerSquareMatrix m) {
                if(m instanceof IntegerTridiagonalMatrix)
                        return subtract((IntegerTridiagonalMatrix)m);
                else
                        return rawSubtract(m);
        }
        private IntegerSquareMatrix rawSubtract(final IntegerSquareMatrix m) {
                if(matrix[1].length==m.rows()) {
                        final int array[][]=new int[matrix[1].length][matrix[1].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=getElement(i,0)-m.getElement(i,0);
                                for(j=1;j<array.length;j++)
                                        array[i][j]=getElement(i,j)-m.getElement(i,j);
                        }
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m an integer tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerTridiagonalMatrix subtract(final IntegerTridiagonalMatrix m) {
                int mRow=matrix[1].length;
                if(mRow==m.rows()) {
                        final IntegerTridiagonalMatrix ans=new IntegerTridiagonalMatrix(mRow);
                        ans.matrix[1][0]=matrix[1][0]-m.getElement(0,0);
                        ans.matrix[2][0]=matrix[2][0]-m.getElement(0,1);
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.matrix[0][i]=matrix[0][i]-m.getElement(i,i-1);
                                ans.matrix[1][i]=matrix[1][i]-m.getElement(i,i);
                                ans.matrix[2][i]=matrix[2][i]-m.getElement(i,i+1);
                        }
                        ans.matrix[0][mRow]=matrix[0][mRow]-m.getElement(mRow,mRow-1);
                        ans.matrix[1][mRow]=matrix[1][mRow]-m.getElement(mRow,mRow);
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x an integer
        * @return an integer tridiagonal matrix
        */
        public IntegerMatrix scalarMultiply(final int x) {
                int mRow=matrix[1].length;
                final IntegerTridiagonalMatrix ans=new IntegerTridiagonalMatrix(mRow);
                ans.matrix[1][0]=x*matrix[1][0];
                ans.matrix[2][0]=x*matrix[2][0];
                mRow--;
                for(int i=1;i<mRow;i++) {
                        ans.matrix[0][i]=x*matrix[0][i];
                        ans.matrix[1][i]=x*matrix[1][i];
                        ans.matrix[2][i]=x*matrix[2][i];
                }
                ans.matrix[0][mRow]=x*matrix[0][mRow];
                ans.matrix[0][mRow]=x*matrix[1][mRow];
                return ans;
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v an integer vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public IntegerVector multiply(final IntegerVector v) {
                int mRow=matrix[1].length;
                if(mRow==v.dimension()) {
                        final int array[]=new int[mRow];
                        array[0]=matrix[1][0]*v.getComponent(0)+matrix[2][0]*v.getComponent(1);
                        mRow--;
                        for(int i=1;i<mRow;i++)
                                array[i]=matrix[0][i]*v.getComponent(i-1)+matrix[1][i]*v.getComponent(i)+matrix[2][i]*v.getComponent(i+1);
                        array[mRow]=matrix[0][mRow]*v.getComponent(mRow-1)+matrix[1][mRow]*v.getComponent(mRow);
                        return new IntegerVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m an integer matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerMatrix multiply(final IntegerMatrix m) {
                if(m instanceof IntegerTridiagonalMatrix)
                        return multiply((IntegerTridiagonalMatrix)m);
                else if(m instanceof IntegerSquareMatrix)
                        return rawMultiply((IntegerSquareMatrix)m);
                else
                        return rawMultiply(m);
        }
        private IntegerMatrix rawMultiply(final IntegerMatrix m) {
                if(matrix[1].length==m.rows()) {
                        int n,k;
                        final int array[][]=new int[matrix[1].length][m.columns()];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array[0].length;k++) {
                                        array[j][k]=getElement(j,0)*m.getElement(0,k);
                                        for(n=1;n<matrix[1].length;n++)
                                                array[j][k]+=getElement(j,n)*m.getElement(n,k);
                                }
                        }
                        return new IntegerMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m an integer square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerSquareMatrix multiply(final IntegerSquareMatrix m) {
                if(m instanceof IntegerTridiagonalMatrix)
                        return multiply((IntegerTridiagonalMatrix)m);
                else
                        return rawMultiply(m);
        }
        private IntegerSquareMatrix rawMultiply(final IntegerSquareMatrix m) {
                if(matrix[1].length==m.rows()) {
                        int n,k;
                        final int array[][]=new int[matrix[1].length][matrix[1].length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array.length;k++) {
                                        array[j][k]=getElement(j,0)*m.getElement(0,k);
                                        for(n=1;n<array.length;n++)
                                                array[j][k]+=getElement(j,n)*m.getElement(n,k);
                                }
                        }
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m an integer tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerSquareMatrix multiply(final IntegerTridiagonalMatrix m) {
                int mRow=matrix[1].length;
                if(mRow==m.rows()) {
                        final int array[][]=new int[mRow][mRow];
                        array[0][0]=matrix[1][0]*m.getElement(0,0)+matrix[2][0]*m.getElement(1,0);
                        array[0][1]=matrix[1][0]*m.getElement(0,1)+matrix[2][0]*m.getElement(1,1);
                        array[0][2]=matrix[2][0]*m.getElement(1,2);
                        if(mRow>3) {
                                array[1][0]=matrix[0][1]*m.getElement(0,0)+matrix[1][1]*m.getElement(1,0);
                                array[1][1]=matrix[0][1]*m.getElement(0,1)+matrix[1][1]*m.getElement(1,1)+matrix[2][1]*m.getElement(2,1);
                                array[1][2]=matrix[1][1]*m.getElement(1,2)+matrix[2][1]*m.getElement(2,2);
                                array[1][3]=matrix[2][1]*m.getElement(2,3);
                        }
                        if(mRow==3) {
                                array[1][0]=matrix[0][1]*m.getElement(0,0)+matrix[1][1]*m.getElement(1,0);
                                array[1][1]=matrix[0][1]*m.getElement(0,1)+matrix[1][1]*m.getElement(1,1)+matrix[2][1]*m.getElement(2,1);
                                array[1][2]=matrix[1][1]*m.getElement(1,2)+matrix[2][1]*m.getElement(2,2);
                        } else if(mRow>4) {
                                for(int i=2;i<mRow-2;i++) {
                                        array[i][i-2]=matrix[0][i]*m.getElement(i-1,i-2);
                                        array[i][i-1]=matrix[0][i]*m.getElement(i-1,i-1)+matrix[1][i]*m.getElement(i,i-1);
                                        array[i][i]=matrix[0][i]*m.getElement(i-1,i)+matrix[1][i]*m.getElement(i,i)+matrix[2][i]*m.getElement(i+1,i);
                                        array[i][i+1]=matrix[1][i]*m.getElement(i,i+1)+matrix[2][i]*m.getElement(i+1,i+1);
                                        array[i][i+2]=matrix[2][i]*m.getElement(i+1,i+2);
                                }
                        }
                        if(mRow>3) {
                                array[mRow-2][mRow-4]=matrix[0][mRow-2]*m.getElement(mRow-3,mRow-4);
                                array[mRow-2][mRow-3]=matrix[0][mRow-2]*m.getElement(mRow-3,mRow-3)+matrix[1][mRow-2]*m.getElement(mRow-2,mRow-3);
                                array[mRow-2][mRow-2]=matrix[0][mRow-2]*m.getElement(mRow-3,mRow-2)+matrix[1][mRow-2]*m.getElement(mRow-2,mRow-2)+matrix[2][mRow-2]*m.getElement(mRow-1,mRow-2);
                                array[mRow-2][mRow-1]=matrix[1][mRow-2]*m.getElement(mRow-2,mRow-1)+matrix[2][mRow-2]*m.getElement(mRow-1,mRow-1);
                        }
                        mRow--;
                        array[mRow][mRow-2]=matrix[0][mRow]*m.getElement(mRow-1,mRow-2);
                        array[mRow][mRow-1]=matrix[0][mRow]*m.getElement(mRow-1,mRow-1)+matrix[1][mRow]*m.getElement(mRow,mRow-1);
                        array[mRow][mRow]=matrix[0][mRow]*m.getElement(mRow-1,mRow)+matrix[1][mRow]*m.getElement(mRow,mRow);
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return an integer tridiagonal matrix
        */
        public Matrix transpose() {
                final IntegerTridiagonalMatrix ans=new IntegerTridiagonalMatrix(matrix[1].length);
                System.arraycopy(matrix[0],1,ans.matrix[2],0,matrix[0].length-1);
                System.arraycopy(matrix[1],0,ans.matrix[1],0,matrix[1].length);
                System.arraycopy(matrix[2],0,ans.matrix[0],1,matrix[2].length-1);
                return ans;
        }

// LU DECOMPOSITION

        /**
        * Returns the LU decomposition of this matrix.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] luDecompose() {
                int i,j,k;
                double tmp;
                final double array[][][]=new double[2][matrix[1].length][matrix[1].length];
                array[0][0][0]=1.0;
                for(i=1;i<matrix[1].length;i++)
                        array[0][i][i]=1.0;
                for(j=0;j<matrix[1].length;j++) {
                        for(i=0;i<=j;i++) {
                                tmp=getElement(i,j);
                                for(k=0;k<i;k++)
                                        tmp-=array[0][i][k]*array[1][k][j];
                                array[1][i][j]=tmp;
                        }
                        for(i=j+1;i<matrix[1].length;i++) {
                                tmp=getElement(i,j);
                                for(k=0;k<j;k++)
                                        tmp-=array[0][i][k]*array[1][k][j];
                                array[0][i][j]=tmp/array[1][j][j];
                        }
                }
                final DoubleSquareMatrix lu[]=new DoubleSquareMatrix[2];
                lu[0]=new DoubleSquareMatrix(array[0]);
                lu[1]=new DoubleSquareMatrix(array[1]);
                return lu;
        }

// CHOLESKY DECOMPOSITION

        /**
        * Returns the Cholesky decomposition of this matrix.
        * Matrix must be symmetric and positive definite.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] choleskyDecompose() {
                int i,j,k;
                double tmp;
                final double array[][][]=new double[2][matrix[1].length][matrix[1].length];
                array[0][0][0]=array[1][0][0]=Math.sqrt(getElement(0,0));
                for(i=1;i<matrix[1].length;i++)
                        array[0][i][0]=array[1][0][i]=getElement(i,0)/array[0][0][0];
                for(j=1;j<matrix[1].length;j++) {
                        tmp=getElement(j,j);
                        for(i=0;i<j;i++)
                                tmp-=array[0][j][i]*array[0][j][i];
                        array[0][j][j]=array[1][j][j]=Math.sqrt(tmp);
                        for(i=j+1;i<matrix[1].length;i++) {
                                tmp=getElement(i,j);
                                for(k=0;k<i;k++)
                                        tmp-=array[0][j][k]*array[1][k][i];
                                array[0][i][j]=array[1][j][i]=tmp/array[1][j][j];
                        }
                }
                final DoubleSquareMatrix lu[]=new DoubleSquareMatrix[2];
                lu[0]=new DoubleSquareMatrix(array[0]);
                lu[1]=new DoubleSquareMatrix(array[1]);
                return lu;
        }
}

