package JSci.maths;

import JSci.GlobalSettings;
import JSci.maths.groups.AbelianGroupMember;
import JSci.maths.fields.FieldMember;

/**
* The MathDouble class encapsulates double numbers.
* @version 1.0
* @author Mark Hale
*/
public final class MathDouble extends Object implements FieldMember, java.io.Serializable {
        private double x;
        /**
        * Constructs a double number.
        */
        public MathDouble(final double num) {
                x=num;
        }
        /**
        * Constructs the double number represented by a string.
        * @param s a string representing a double number
        * @exception NumberFormatException if the string does not contain a parsable number.
        */
        public MathDouble(final String s) throws NumberFormatException {
                x=Double.valueOf(s).doubleValue();
        }
        /**
        * Compares two double numbers for equality.
        * @param obj a double number
        */
        public boolean equals(Object obj) {
                if(obj!=null && (obj instanceof MathDouble)) {
                        return Math.abs(x-((MathDouble)obj).value())<=GlobalSettings.ZERO_TOL;
                } else
                        return false;
        }
        /**
        * Returns a string representing the value of this double number.
        */
        public String toString() {
                return Double.toString(x);
        }
        /**
        * Returns the double value.
        */
        public double value() {
                return x;
        }
        /**
        * Returns true if this number is NaN.
        */
        public boolean isNaN() {
                return (x==Double.NaN);
        }
        /**
        * Returns true if this number is infinite.
        */
        public boolean isInfinite() {
                return (x==Double.POSITIVE_INFINITY) || (x==Double.NEGATIVE_INFINITY);
        }
        /**
        * Returns the negative of this number.
        */
        public AbelianGroupMember negate() {
                return new MathDouble(-x);
        }
        /**
        * Returns the inverse of this number.
        */
        public FieldMember inverse() {
                return new MathDouble(1.0/x);
        }
        /**
        * Returns the addition of this number and another.
        */
        public AbelianGroupMember add(final AbelianGroupMember n) {
                if(n instanceof MathDouble)
                        return add((MathDouble)n);
                else if(n instanceof MathInteger)
                        return add(new MathDouble(((MathInteger)n).value()));
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the addition of this double number and another.
        */
        public MathDouble add(final MathDouble n) {
                return new MathDouble(x+n.value());
        }
        /**
        * Returns the subtraction of this number and another.
        */
        public AbelianGroupMember subtract(final AbelianGroupMember n) {
                if(n instanceof MathDouble)
                        return subtract((MathDouble)n);
                else if(n instanceof MathInteger)
                        return subtract(new MathDouble(((MathInteger)n).value()));
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the subtraction of this double number and another.
        */
        public MathDouble subtract(final MathDouble n) {
                return new MathDouble(x-n.value());
        }
        /**
        * Returns the multiplication of this number and another.
        */
        public RingMember multiply(final RingMember n) {
                if(n instanceof MathDouble)
                        return multiply((MathDouble)n);
                else if(n instanceof MathInteger)
                        return multiply(new MathDouble(((MathInteger)n).value()));
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the multiplication of this double number and another.
        */
        public MathDouble multiply(final MathDouble n) {
                return new MathDouble(x*n.value());
        }
        /**
        * Returns the division of this number and another.
        */
        public FieldMember divide(final FieldMember n) {
                if(n instanceof MathDouble)
                        return divide((MathDouble)n);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the division of this double number and another.
        */
        public MathDouble divide(final MathDouble n) {
                return new MathDouble(x/n.value());
        }
}

