package JSci.maths;

import JSci.GlobalSettings;
import JSci.maths.groups.AbelianGroupMember;
import JSci.maths.fields.FieldMember;

/**
* The Quaternion class encapsulates quaternions.
* @version 1.0
* @author Mark Hale
*/
public final class Quaternion extends Object implements FieldMember, java.io.Serializable {
        private double re;
        private Double3Vector im;
        /**
        * Constructs a quaternion.
        */
        public Quaternion(final double real,final Double3Vector imag) {
                re=real;
                im=imag;
        }
        /**
        * Constructs the quaternion q<SUB>0</SUB>+iq<SUB>1</SUB>+jq<SUB>2</SUB>+kq<SUB>3</SUB>.
        */
        public Quaternion(final double q0,final double q1,final double q2,final double q3) {
                re=q0;
                im=new Double3Vector(q1,q2,q3);
        }
        /**
        * Compares two quaternions for equality.
        * @param obj a quaternion
        */
        public boolean equals(Object obj) {
                if(obj!=null && (obj instanceof Quaternion)) {
                        final Quaternion q=(Quaternion)obj;
                        return Math.abs(re-q.re)<=GlobalSettings.ZERO_TOL &&
                                im.equals(q.im);
                } else
                        return false;
        }
        /**
        * Returns a string representing the value of this quaternion.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer();
                buf.append(re);
                if(im.vector[0]>=0.0)
                        buf.append("+");
                buf.append(im.vector[0]);
                buf.append("i");
                if(im.vector[1]>=0.0)
                        buf.append("+");
                buf.append(im.vector[1]);
                buf.append("j");
                if(im.vector[2]>=0.0)
                        buf.append("+");
                buf.append(im.vector[2]);
                buf.append("k");
                return buf.toString();
        }
        /**
        * Returns a hashcode for this quaternion.
        */
        public int hashCode() {
                return (int)(Math.exp(mod()));
        }
        /**
        * Returns true if either the real or imaginary part is NaN.
        */
        public boolean isNaN() {
                return (re==Double.NaN) || (im.vector[0]==Double.NaN) ||
                (im.vector[1]==Double.NaN) || (im.vector[2]==Double.NaN);
        }
        /**
        * Returns true if either the real or imaginary part is infinite.
        */
        public boolean isInfinite() {
                return (re==Double.POSITIVE_INFINITY) || (re==Double.NEGATIVE_INFINITY)
                        || (im.vector[0]==Double.POSITIVE_INFINITY) || (im.vector[0]==Double.NEGATIVE_INFINITY)
                        || (im.vector[1]==Double.POSITIVE_INFINITY) || (im.vector[1]==Double.NEGATIVE_INFINITY)
                        || (im.vector[2]==Double.POSITIVE_INFINITY) || (im.vector[2]==Double.NEGATIVE_INFINITY);
        }
        /**
        * Returns the real part of this quaternion.
        */
        public double real() {
                return re;
        }
        /**
        * Returns the imaginary part of this quaternion.
        */
        public Double3Vector imag() {
                return im;
        }
        /**
        * Returns the modulus of this quaternion.
        */
        public double mod() {
                final double norm=im.norm();
                if(re==0.0 && norm==0.0)
                        return 0.0;
                final double reAbs=Math.abs(re),normAbs=Math.abs(norm);
                if(reAbs<normAbs)
                        return normAbs*Math.sqrt(1.0+(re/norm)*(re/norm));
                else
                        return reAbs*Math.sqrt(1.0+(norm/re)*(norm/re));
        }

//============
// OPERATIONS
//============

        /**
        * Returns the negative of this quaternion.
        */
        public AbelianGroupMember negate() {
                return new Quaternion(-re,-im.vector[0],-im.vector[1],-im.vector[2]);
        }
        /**
        * Returns the inverse of this quaternion.
        */
        public FieldMember inverse() {
                final double thisMod=mod();
                return conjugate().divide(thisMod*thisMod);
        }
        /**
        * Returns the conjugate of this quaternion.
        */
        public Quaternion conjugate() {
                return new Quaternion(re,-im.vector[0],-im.vector[1],-im.vector[2]);
        }

// ADDITION

        /**
        * Returns the addition of this number and another.
        */
        public AbelianGroupMember add(final AbelianGroupMember x) {
                if(x instanceof Quaternion)
                        return add((Quaternion)x);
                else if(x instanceof MathDouble)
                        return addReal(((MathDouble)x).value());
                else if(x instanceof MathInteger)
                        return addReal(((MathInteger)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the addition of this quaternion and another.
        * @param q a quaternion
        */
        public Quaternion add(final Quaternion q) {
                return new Quaternion(re+q.re,im.add(q.im));
        }
        /**
        * Returns the addition of this quaternion with a real part.
        * @param real a real part
        */
        public Quaternion addReal(final double real) {
                return new Quaternion(re+real,im);
        }
        /**
        * Returns the addition of this quaternion with an imaginary part.
        * @param imag an imaginary part
        */
        public Quaternion addImag(final Double3Vector imag) {
                return new Quaternion(re,im.add(imag));
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this number and another.
        */
        public AbelianGroupMember subtract(final AbelianGroupMember x) {
                if(x instanceof Quaternion)
                        return subtract((Quaternion)x);
                else if(x instanceof MathDouble)
                        return subtractReal(((MathDouble)x).value());
                else if(x instanceof MathInteger)
                        return subtractReal(((MathInteger)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the subtraction of this quaternion by another.
        * @param q a quaternion
        */
        public Quaternion subtract(final Quaternion q) {
                return new Quaternion(re-q.re,im.subtract(q.im));
        }
        /**
        * Returns the subtraction of this quaternion by a real part.
        * @param real a real part
        */
        public Quaternion subtractReal(final double real) {
                return new Quaternion(re-real,im);
        }
        /**
        * Returns the subtraction of this quaternion by an imaginary part.
        * @param imag an imaginary part
        */
        public Quaternion subtractImag(final Double3Vector imag) {
                return new Quaternion(re,im.subtract(imag));
        }

// MULTIPLICATION

        /**
        * Returns the multiplication of this number and another.
        */
        public RingMember multiply(final RingMember x) {
                if(x instanceof Quaternion)
                        return multiply((Quaternion)x);
                else if(x instanceof MathDouble)
                        return multiply(((MathDouble)x).value());
                else if(x instanceof MathInteger)
                        return multiply(((MathInteger)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the multiplication of this quaternion and another.
        * @param q a quaternion
        */
        public Quaternion multiply(final Quaternion q) {
                return new Quaternion(
                        re*q.re-im.scalarProduct(q.im),
                        (Double3Vector)(q.im.scalarMultiply(re).add(im.scalarMultiply(q.re)).add(im.multiply(q.im)))
                );
        }
        /**
        * Returns the multiplication of this quaternion by a scalar.
        * @param x a real number
        */
        public Quaternion multiply(final double x) {
                return new Quaternion(x*re,(Double3Vector)im.scalarMultiply(x));
        }

// DIVISION

        /**
        * Returns the division of this number and another.
        */
        public FieldMember divide(final FieldMember x) {
                if(x instanceof Quaternion)
                        return divide((Quaternion)x);
                else if(x instanceof MathDouble)
                        return divide(((MathDouble)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the division of this quaternion by another.
        * @param q a quaternion
        * @exception ArithmeticException If divide by zero.
        */
        public Quaternion divide(final Quaternion q) {
                final double qMod=q.mod();
                return multiply(q.conjugate()).divide(qMod*qMod);
        }
        /**
        * Returns the division of this quaternion by a scalar.
        * @param x a real number
        * @exception ArithmeticException If divide by zero.
        */
        public Quaternion divide(final double x) {
                return new Quaternion(re/x,(Double3Vector)im.scalarDivide(x));
        }
}

