package JSci.maths.groups;

import JSci.maths.*;

/**
* The LieGroup class provides an encapsulation for Lie groups.
* Elements are represented by complex matrices, and are limited
* to being near the identity.
* @version 1.2
* @author Mark Hale
*/
public class LieGroup extends Object implements Group {
        private ComplexSquareMatrix generators[];
        private ComplexSquareMatrix identityMatrix;
        /**
        * Constructs a Lie group from a Lie algebra.
        * @param gens the group generators
        */
        public LieGroup(ComplexSquareMatrix gens[]) {
                generators=gens;
                identityMatrix=ComplexDiagonalMatrix.identity(generators[0].rows());
        }
        /**
        * Returns the dimension of the group.
        */
        public final int dimension() {
                return generators.length;
        }
        /**
        * Returns an element near the identity.
        * @param v a small element from the Lie algebra
        */
        public ComplexSquareMatrix getElement(DoubleVector v) {
                if(generators.length!=v.dimension())
                        throw new IllegalArgumentException("The vector should match the generators.");
                ComplexMatrix phase=generators[0].scalarMultiply(v.getComponent(0));
                for(int i=1;i<generators.length;i++)
                        phase=phase.add(generators[i].scalarMultiply(v.getComponent(i)));
                return (ComplexSquareMatrix)identityMatrix.add(phase.scalarMultiply(Complex.I));
        }
        /**
        * The group composition law.
        * @param a a group element
        * @param b a group element
        */
        public Object compose(Object a, Object b) {
                if(!(a instanceof ComplexSquareMatrix) || !(b instanceof ComplexSquareMatrix))
                        throw new IllegalArgumentException("Must be a complex square matrix.");
                return ((ComplexSquareMatrix)a).multiply((ComplexSquareMatrix)b);
        }
        /**
        * Returns the identity element.
        */
        public Object identity() {
                return identityMatrix;
        }
        /**
        * Returns the inverse element.
        */
        public Object inverse(Object a) {
                if(!(a instanceof ComplexSquareMatrix))
                        throw new IllegalArgumentException("Must be a complex square matrix.");
                return ((ComplexSquareMatrix)a).inverse();
        }
        /**
        * Returns true if the element is the identity element of this group.
        * @param a a group element
        */
        public final boolean isIdentity(Object a) {
                return identityMatrix.equals(a);
        }
        /**
        * Returns true if one element is the inverse of the other.
        * @param a a group element
        * @param b a group element
        */
        public final boolean isInverse(Object a, Object b) {
                return isIdentity(compose(a,b));
        }
}

