package JSci.maths.statistics;

import JSci.maths.SpecialMath;

/**
* The GammaDistribution class provides an object for encapsulating gamma distributions.
* @version 1.0
* @author Jaco van Kooten
*/
public final class GammaDistribution extends ProbabilityDistribution {
        private double a;

        /**
        * Constructs a gamma distribution.
        * @param shape the shape parameter
        */
        public GammaDistribution(double shape) {
                if(shape<=0.0)
                        throw new OutOfRangeException("The shape parameter should be (strictly) positive.");
                a=shape;
        }
        /**
        * Returns the shape parameter.
        */
        public double getShapeParameter() {
                return a;
        }
        /**
        * Probability density function of a gamma distribution.
        * @return the probability that a stochastic variable x has the value X, i.e. P(x=X)
        */
        public double probability(double X) {
                checkRange(X,0.0,Double.MAX_VALUE);
                if(X==0.0)
                        return 0.0;
                else
                        return Math.exp(-SpecialMath.logGamma(a)-X+(a-1)*Math.log(X));
        }
        /**
        * Cumulative gamma distribution function.
	* @return the probability that a stochastic variable x is less then X, i.e. P(x&lt;X)
        */
        public double cumulative(double X) {
                checkRange(X,0.0,Double.MAX_VALUE);
                return SpecialMath.incompleteGamma(a,X);
        }
        /**
	* Inverse of the cumulative gamma distribution function.
        * @return the value X for which P(x&lt;X)
        */
        public double inverse(double probability) {
                checkRange(probability);
                if(probability==0.0)
                        return 0.0;
                if(probability==1.0)
                        return Double.MAX_VALUE;
                return findRoot(probability, a, 0.0, Double.MAX_VALUE);
        }
}

