package JSci.io;

import JSci.maths.*;
import org.w3c.dom.*;
import org.apache.xerces.parsers.DOMParser;

/**
* The MathMLParser class will parse a MathML document into JSci objects.
* @version 0.5
* @author Mark Hale
*/
public final class MathMLParser extends DOMParser {
        private static final int INTEGER=0;
        private static final int DOUBLE=1;
        private static final int COMPLEX=2;
        private int setCounter=1;
        /**
        * Constructs a MathMLParser.
        */
        public MathMLParser() {}
        /**
        * Returns the JSci objects from the parsed document.
        */
        public Object[] getJSciObjects() {
                return parseMATH(fDocument.getDocumentElement());
        }
        /**
        * Parses the &lt;apply&gt; tag.
        */
        private Object[] parseMATH(Node n) {
                int len=0;
                final NodeList nl=n.getChildNodes();
                final Object objList[]=new Object[nl.getLength()];
                Object obj;
                for(int i=0;i<objList.length;i++) {
                        obj=processNode(nl.item(i));
                        if(obj!=null) {
                                objList[len]=obj;
                                len++;
                        }
                }
                final Object parseList[]=new Object[len];
                System.arraycopy(objList,0,parseList,0,len);
                return parseList;
        }
        /**
        * Processes a node.
        */
        private Object processNode(Node n) {
                final String nodeName=n.getNodeName();
                if(nodeName.equals("apply"))
                        return parseAPPLY(n);
                else if(nodeName.equals("reln"))
                        return parseRELN(n);
                else if(nodeName.equals("cn"))
                        return parseCN(n);
                else if(nodeName.equals("ci"))
                        return parseCI(n);
                else if(nodeName.equals("vector"))
                        return parseVECTOR(n);
                else if(nodeName.equals("matrix"))
                        return parseMATRIX(n);
                else if(nodeName.equals("set"))
                        return parseSET(n);
                else
                        return null;
        }
        /**
        * Parses &lt;apply&gt; tags.
        */
        private MathMLExpression parseAPPLY(Node n) {
                final MathMLExpression exp=new MathMLExpression();
                final NodeList nl=n.getChildNodes();
                Object obj;
                int i;
                for(i=0;nl.item(i) instanceof Text;i++)
                        ;
                exp.setOperation(nl.item(i).getNodeName());
                for(;i<nl.getLength();i++) {
                        obj=processNode(nl.item(i));
                        if(obj!=null)
                                exp.addArgument(obj);
                }
                return exp;
        }
        /**
        * Parses &lt;reln&gt; tags.
        */
        private MathMLRelation parseRELN(Node n) {
                final MathMLRelation exp=new MathMLRelation();
                final NodeList nl=n.getChildNodes();
                Object obj;
                int i;
                for(i=0;nl.item(i) instanceof Text;i++)
                        ;
                exp.setRelation(nl.item(i).getNodeName());
                for(;i<nl.getLength();i++) {
                        obj=processNode(nl.item(i));
                        if(obj!=null)
                                exp.addArgument(obj);
                }
                return exp;
        }
        /**
        * Parses &lt;cn&gt; tags.
        */
        private RingMember parseCN(Node n) {
                final NamedNodeMap attr=n.getAttributes();
// support only base 10
                if(!attr.getNamedItem("base").getNodeValue().equals("10"))
                        return null;
// default type="real"
                if(attr.getNamedItem("type")==null)
                        return new MathDouble(n.getFirstChild().getNodeValue());
                final String attrType=attr.getNamedItem("type").getNodeValue();
                if(attrType.equals("integer")) {
                        return new MathInteger(n.getFirstChild().getNodeValue());
                } else if(attrType.equals("real")) {
                        return new MathDouble(n.getFirstChild().getNodeValue());
                } else if(attrType.equals("rational")) {
                        final Node num=n.getFirstChild();
                        final Node denom=num.getNextSibling().getNextSibling();
                        return new MathDouble(num.getNodeValue()).divide(new MathDouble(denom.getNodeValue()));
                } else if(attrType.equals("complex-cartesian")) {
                        final Node re=n.getFirstChild();
                        final Node im=re.getNextSibling().getNextSibling();
                        return new Complex(
                                new Double(re.getNodeValue()).doubleValue(),
                                new Double(im.getNodeValue()).doubleValue()
                        );
                } else if(attrType.equals("complex-polar")) {
                        final Node mod=n.getFirstChild();
                        final Node arg=mod.getNextSibling().getNextSibling();
                        return Complex.polar(
                                new Double(mod.getNodeValue()).doubleValue(),
                                new Double(arg.getNodeValue()).doubleValue()
                        );
                } else
                        return null;
        }
        /**
        * Parses &lt;ci&gt; tags.
        */
        private String parseCI(Node n) {
                return n.getFirstChild().getNodeValue();
        }
        /**
        * Parses &lt;vector&gt; tags.
        */
        private MathVector parseVECTOR(Node n) {
                int len=0,type=INTEGER;
                final NodeList nl=n.getChildNodes();
                final RingMember num[]=new RingMember[nl.getLength()];
                for(int i=0;i<num.length;i++) {
                        if(nl.item(i).getNodeName().equals("cn")) {
                                num[len]=parseCN(nl.item(i));
                                if(num[len]!=null) {
// work out number format needed
                                        if(num[len] instanceof MathDouble && type<DOUBLE)
                                                type=DOUBLE;
                                        else if(num[len] instanceof Complex && type<COMPLEX)
                                                type=COMPLEX;
                                        len++;
                                }
                        }
                }
// output to JSci objects
                if(type==INTEGER) {
                        final int array[]=new int[len];
                        for(int i=0;i<array.length;i++)
                                array[i]=((MathInteger)num[i]).value();
                        return new IntegerVector(array);
                } else if(type==DOUBLE) {
                        final double array[]=new double[len];
                        for(int i=0;i<array.length;i++) {
                                if(num[i] instanceof MathInteger)
                                        array[i]=((MathInteger)num[i]).value();
                                else
                                        array[i]=((MathDouble)num[i]).value();
                        }
                        return new DoubleVector(array);
                } else {
                        final Complex array[]=new Complex[len];
                        for(int i=0;i<array.length;i++) {
                                if(num[i] instanceof MathInteger)
                                        array[i]=new Complex(((MathInteger)num[i]).value(),0.0);
                                else if(num[i] instanceof MathDouble)
                                        array[i]=new Complex(((MathDouble)num[i]).value(),0.0);
                                else
                                        array[i]=(Complex)num[i];
                        }
                        return new ComplexVector(array);
                }
        }
        /**
        * Parses &lt;matrix&gt; tags.
        */
        private Matrix parseMATRIX(Node n) {
                int rows=0,cols=Integer.MAX_VALUE;
                final NodeList nl=n.getChildNodes();
                final RingMember num[][]=new RingMember[nl.getLength()][];
                for(int i=0;i<num.length;i++) {
                        if(nl.item(i).getNodeName().equals("matrixrow")) {
                                num[rows]=parseMATRIXROW(nl.item(i));
                                if(num[rows].length<cols)
                                        cols=num[rows].length;
                               rows++;
                        }
                }
// work out number format needed
                int type=INTEGER;
                for(int j,i=0;i<rows;i++) {
                        for(j=0;j<cols;j++) {
                                if(num[i][j] instanceof MathDouble && type<DOUBLE)
                                        type=DOUBLE;
                                else if(num[i][j] instanceof Complex && type<COMPLEX)
                                        type=COMPLEX;
                        }
                }
// output to JSci objects
                if(type==INTEGER) {
                        final int array[][]=new int[rows][cols];
                        for(int j,i=0;i<rows;i++) {
                                for(j=0;j<cols;j++)
                                        array[i][j]=((MathInteger)num[i][j]).value();
                        }
                        if(rows==cols)
                                return new IntegerSquareMatrix(array);
                        else
                                return new IntegerMatrix(array);
                } else if(type==DOUBLE) {
                        final double array[][]=new double[rows][cols];
                        for(int j,i=0;i<rows;i++) {
                                for(j=0;j<cols;j++) {
                                        if(num[i][j] instanceof MathInteger)
                                                array[i][j]=((MathInteger)num[i][j]).value();
                                        else
                                                array[i][j]=((MathDouble)num[i][j]).value();
                                }
                        }
                        if(rows==cols)
                                return new DoubleSquareMatrix(array);
                        else
                                return new DoubleMatrix(array);
                } else {
                        final Complex array[][]=new Complex[rows][cols];
                        for(int j,i=0;i<rows;i++) {
                                for(j=0;j<cols;j++) {
                                        if(num[i][j] instanceof MathInteger)
                                                array[i][j]=new Complex(((MathInteger)num[i][j]).value(),0.0);
                                        else if(num[i][j] instanceof MathDouble)
                                                array[i][j]=new Complex(((MathDouble)num[i][j]).value(),0.0);
                                        else
                                                array[i][j]=(Complex)num[i][j];
                                }
                        }
                        if(rows==cols)
                                return new ComplexSquareMatrix(array);
                        else
                                return new ComplexMatrix(array);
                }

        }
        /**
        * Parses &lt;matrixrow&gt; tags.
        */
        private RingMember[] parseMATRIXROW(Node n) {
                int len=0;
                final NodeList nl=n.getChildNodes();
                final RingMember num[]=new RingMember[nl.getLength()];
                for(int i=0;i<num.length;i++) {
                        if(nl.item(i).getNodeName().equals("cn")) {
                                num[len]=parseCN(nl.item(i));
                                if(num[len]!=null)
                                        len++;
                        }
                }
                final RingMember row[]=new RingMember[len];
                System.arraycopy(num,0,row,0,len);
                return row;
        }
        /**
        * Parses &lt;set&gt; tags.
        */
        private DiscreteSet parseSET(Node n) {
                int len=0;
                final NodeList nl=n.getChildNodes();
                final Object objs[]=new Object[nl.getLength()];
                for(int i=0;i<objs.length;i++) {
                        if(nl.item(i).getNodeName().equals("ci")) {
                                objs[len]=parseCI(nl.item(i));
                                len++;
                        }
                }
                final Object elements[]=new Object[len];
                System.arraycopy(objs,0,elements,0,len);
// output to JSci objects
                return new DiscreteSet("Parsed MathML set "+setCounter++,elements);
        }
}

