package JSci.maths;

/**
* The ComplexTridiagonalMatrix class provides an object for encapsulating tridiagonal matrices containing complex numbers.
* Uses compressed diagonal storage.
* @version 2.1
* @author Mark Hale
*/
public class ComplexTridiagonalMatrix extends ComplexSquareMatrix {
//
// example: element storage for a 3x3 tridiagonal matrix
//
// matrix[1][0] matrix[2][0]
// matrix[0][1] matrix[1][1] matrix[2][1]
//              matrix[0][2] matrix[1][2]
//
        /**
        * Storage format identifier.
        */
        protected final static int TRIDIAGONAL=3;
        protected final static int storageFormat=TRIDIAGONAL;
        /**
        * Constructs a matrix.
        */
        protected ComplexTridiagonalMatrix() {
                super();
        }
        /**
        * Constructs an empty matrix.
        * @param size the number of rows/columns
        */
        public ComplexTridiagonalMatrix(final int size) {
                this();
                matrix=new Complex[3][size];
        }
        /**
        * Constructs a matrix from an array.
        * Any non-tridiagonal elements in the array are ignored.
        * @param array an assigned value
        * @exception MatrixDimensionException If the array is not square.
        */
        public ComplexTridiagonalMatrix(final Complex array[][]) {
                this(array.length);
                if(array.length==array[0].length) {
                        matrix[1][0]=array[0][0];
                        matrix[2][0]=array[0][1];
                        int i=1;
                        for(;i<array.length-1;i++) {
                                matrix[0][i]=array[i][i-1];
                                matrix[1][i]=array[i][i];
                                matrix[2][i]=array[i][i+1];
                        }
                        matrix[0][i]=array[i][i-1];
                        matrix[1][i]=array[i][i];
                } else {
                        matrix=null;
                        throw new MatrixDimensionException("The array is not square.");
                }
        }
        /**
        * Compares two complex tridiagonal matrices for equality.
        * @param m a complex tridiagonal matrix
        */
        public boolean equals(Object m) {
                if(m!=null && (m instanceof ComplexTridiagonalMatrix) &&
                matrix[1].length==((ComplexTridiagonalMatrix)m).rows()) {
                        final ComplexTridiagonalMatrix ctm=(ComplexTridiagonalMatrix)m;
                        if(!matrix[1][0].equals(ctm.getElement(0,0)))
                                return false;
                        if(!matrix[2][0].equals(ctm.getElement(0,1)))
                                return false;
                        int i=1;
                        for(;i<matrix[1].length-1;i++) {
                                if(!matrix[0][i].equals(ctm.getElement(i,i-1)))
                                        return false;
                                if(!matrix[1][i].equals(ctm.getElement(i,i)))
                                        return false;
                                if(!matrix[2][i].equals(ctm.getElement(i,i+1)))
                                        return false;
                        }
                        if(!matrix[0][i].equals(ctm.getElement(i,i-1)))
                                return false;
                        if(!matrix[1][i].equals(ctm.getElement(i,i)))
                                return false;
                        return true;
                } else
                        return false;
        }
        /**
        * Returns a string representing this matrix.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer(matrix.length*matrix[1].length);
                for(int j,i=0;i<rows();i++) {
                        for(j=0;j<columns();j++) {
                                buf.append(getElement(i,j).toString());
                                buf.append(' ');
                        }
                        buf.append('\n');
                }
                return buf.toString();
        }
        /**
        * Returns the real part of this complex matrix.
        * @return a double tridiagonal matrix
        */
        public DoubleMatrix real() {
                final DoubleTridiagonalMatrix m=new DoubleTridiagonalMatrix(matrix[1].length);
                m.matrix[1][0]=matrix[1][0].real();
                m.matrix[2][0]=matrix[2][0].real();
                int i=1;
                for(;i<matrix[1].length-1;i++) {
                        m.matrix[0][i]=matrix[0][i].real();
                        m.matrix[1][i]=matrix[1][i].real();
                        m.matrix[2][i]=matrix[2][i].real();
                }
                m.matrix[0][i]=matrix[0][i].real();
                m.matrix[1][i]=matrix[1][i].real();
                return m;
        }
        /**
        * Returns the imaginary part of this complex matrix.
        * @return a double tridiagonal matrix
        */
        public DoubleMatrix imag() {
                final DoubleTridiagonalMatrix m=new DoubleTridiagonalMatrix(matrix[1].length);
                m.matrix[1][0]=matrix[1][0].imag();
                m.matrix[2][0]=matrix[2][0].imag();
                int i=1;
                for(;i<matrix[1].length-1;i++) {
                        m.matrix[0][i]=matrix[0][i].imag();
                        m.matrix[1][i]=matrix[1][i].imag();
                        m.matrix[2][i]=matrix[2][i].imag();
                }
                m.matrix[0][i]=matrix[0][i].imag();
                m.matrix[1][i]=matrix[1][i].imag();
                return m;
        }
        /**
        * Returns an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public Complex getElement(final int i, final int j) {
                if(i>=0 && i<matrix[1].length && j>=0 && j<matrix[1].length) {
                        if(j==i-1)
                                return matrix[0][i];
                        else if(j==i)
                                return matrix[1][i];
                        else if(j==i+1)
                                return matrix[2][i];
                        else
                                return Complex.ZERO;
                } else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Sets the value of an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @param z a complex number
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public void setElement(final int i, final int j, final Complex z) {
                if(i>=0 && i<matrix[1].length && j>=0 && j<matrix[1].length) {
                        if(j==i-1)
                                matrix[0][i]=z;
                        else if(j==i)
                                matrix[1][i]=z;
                        else if(j==i+1)
                                matrix[2][i]=z;
                        else
                                throw new MatrixDimensionException("Invalid element.");
                } else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Returns the trace.
        */
        public Complex trace() {
                double trRe=matrix[0][0].real();
                double trIm=matrix[0][0].imag();
                for(int i=1;i<matrix[1].length;i++) {
                        trRe+=matrix[0][i].real();
                        trIm+=matrix[0][i].imag();
                }
                return new Complex(trRe,trIm);
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public double infNorm() {
                double result=matrix[1][0].mod()+matrix[2][0].mod();
                double tmpResult;
                int i=1;
                for(;i<matrix[1].length-1;i++) {
                        tmpResult=matrix[0][i].mod()+matrix[1][i].mod()+matrix[2][i].mod();
                        if(tmpResult>result)
                                result=tmpResult;
                }
                tmpResult=matrix[0][i].mod()+matrix[1][i].mod();
                if(tmpResult>result)
                        result=tmpResult;
                return result;
        }
        /**
        * Returns the Frobenius norm.
        * @author Taber Smith
        */
        public double frobeniusNorm() {
                double result=matrix[1][0].mod()*matrix[1][0].mod()+matrix[2][0].mod()*matrix[2][0].mod();
                int i=1;
                for(;i<matrix[1].length-1;i++)
                        result+=matrix[0][i].mod()*matrix[0][i].mod()+matrix[1][i].mod()*matrix[1][i].mod()+matrix[2][i].mod()*matrix[2][i].mod();
                result+=matrix[0][i].mod()*matrix[0][i].mod()+matrix[1][i].mod()*matrix[1][i].mod();
                return Math.sqrt(result);
        }
        /**
        * Returns the number of rows.
        */
        public int rows() {
                return matrix[1].length;
        }
        /**
        * Returns the number of columns.
        */
        public int columns() {
                return matrix[1].length;
        }

//============
// OPERATIONS
//============

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        * @param m a complex matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexMatrix add(final ComplexMatrix m) {
                if(m instanceof ComplexTridiagonalMatrix)
                        return add((ComplexTridiagonalMatrix)m);
                else if(m instanceof ComplexSquareMatrix)
                        return rawAdd((ComplexSquareMatrix)m);
                else
                        return rawAdd(m);
        }
        private ComplexMatrix rawAdd(final ComplexMatrix m) {
                if(matrix[1].length==m.rows() && matrix[1].length==m.columns()) {
                        final Complex array[][]=new Complex[matrix[1].length][matrix[1].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=getElement(i,0).add(m.getElement(i,0));
                                for(j=1;j<array.length;j++)
                                        array[i][j]=getElement(i,j).add(m.getElement(i,j));
                        }
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a complex square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexSquareMatrix add(final ComplexSquareMatrix m) {
                if(m instanceof ComplexTridiagonalMatrix)
                        return add((ComplexTridiagonalMatrix)m);
                else
                        return rawAdd(m);
        }
        private ComplexSquareMatrix rawAdd(final ComplexSquareMatrix m) {
                if(matrix[1].length==m.rows()) {
                        final Complex array[][]=new Complex[matrix[1].length][matrix[1].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=getElement(i,0).add(m.getElement(i,0));
                                for(j=1;j<array.length;j++)
                                        array[i][j]=getElement(i,j).add(m.getElement(i,j));
                        }
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a complex tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexTridiagonalMatrix add(final ComplexTridiagonalMatrix m) {
                int mRow=matrix[1].length;
                if(mRow==m.rows()) {
                        final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(mRow);
                        ans.matrix[1][0]=matrix[1][0].add(m.getElement(0,0));
                        ans.matrix[2][0]=matrix[2][0].add(m.getElement(0,1));
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.matrix[0][i]=matrix[0][i].add(m.getElement(i,i-1));
                                ans.matrix[1][i]=matrix[1][i].add(m.getElement(i,i));
                                ans.matrix[2][i]=matrix[2][i].add(m.getElement(i,i+1));
                        }
                        ans.matrix[0][mRow]=matrix[0][mRow].add(m.getElement(mRow,mRow-1));
                        ans.matrix[1][mRow]=matrix[1][mRow].add(m.getElement(mRow,mRow));
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix and another.
        * @param m a complex matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexMatrix subtract(final ComplexMatrix m) {
                if(m instanceof ComplexTridiagonalMatrix)
                        return subtract((ComplexTridiagonalMatrix)m);
                else if(m instanceof ComplexSquareMatrix)
                        return rawSubtract((ComplexSquareMatrix)m);
                else
                        return rawSubtract(m);
        }
        private ComplexMatrix rawSubtract(final ComplexMatrix m) {
                if(matrix[1].length==m.rows() && matrix[1].length==m.columns()) {
                        final Complex array[][]=new Complex[matrix[1].length][matrix[1].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=getElement(i,0).subtract(m.getElement(i,0));
                                for(j=1;j<array.length;j++)
                                        array[i][j]=getElement(i,j).subtract(m.getElement(i,j));
                        }
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a complex square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexSquareMatrix subtract(final ComplexSquareMatrix m) {
                if(m instanceof ComplexTridiagonalMatrix)
                        return subtract((ComplexTridiagonalMatrix)m);
                else
                        return rawSubtract(m);
        }
        private ComplexSquareMatrix rawSubtract(final ComplexSquareMatrix m) {
                if(matrix[1].length==m.rows()) {
                        final Complex array[][]=new Complex[matrix[1].length][matrix[1].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=getElement(i,0).subtract(m.getElement(i,0));
                                for(j=1;j<array.length;j++)
                                        array[i][j]=getElement(i,j).subtract(m.getElement(i,j));
                        }
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix and another.
        * @param m a complex tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexTridiagonalMatrix subtract(final ComplexTridiagonalMatrix m) {
                int mRow=matrix[1].length;
                if(mRow==m.rows()) {
                        final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(mRow);
                        ans.matrix[1][0]=matrix[1][0].subtract(m.getElement(0,0));
                        ans.matrix[2][0]=matrix[2][0].subtract(m.getElement(0,1));
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.matrix[0][i]=matrix[0][i].subtract(m.getElement(i,i-1));
                                ans.matrix[1][i]=matrix[1][i].subtract(m.getElement(i,i));
                                ans.matrix[2][i]=matrix[2][i].subtract(m.getElement(i,i+1));
                        }
                        ans.matrix[0][mRow]=matrix[0][mRow].subtract(m.getElement(mRow,mRow-1));
                        ans.matrix[1][mRow]=matrix[1][mRow].subtract(m.getElement(mRow,mRow));
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param z a complex number
        * @return a complex tridiagonal matrix
        */
        public ComplexMatrix scalarMultiply(final Complex z) {
                int mRow=matrix[1].length;
                final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(mRow);
                ans.matrix[1][0]=z.multiply(matrix[1][0]);
                ans.matrix[2][0]=z.multiply(matrix[2][0]);
                mRow--;
                for(int i=1;i<mRow;i++) {
                        ans.matrix[0][i]=z.multiply(matrix[0][i]);
                        ans.matrix[1][i]=z.multiply(matrix[1][i]);
                        ans.matrix[2][i]=z.multiply(matrix[2][i]);
                }
                ans.matrix[0][mRow]=z.multiply(matrix[0][mRow]);
                ans.matrix[1][mRow]=z.multiply(matrix[1][mRow]);
                return ans;
        }
        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x a double
        * @return a complex tridiagonal matrix
        */
        public ComplexMatrix scalarMultiply(final double x) {
                int mRow=matrix[1].length;
                final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(mRow);
                ans.matrix[1][0]=matrix[1][0].multiply(x);
                ans.matrix[2][0]=matrix[2][0].multiply(x);
                mRow--;
                for(int i=1;i<mRow;i++) {
                        ans.matrix[0][i]=matrix[0][i].multiply(x);
                        ans.matrix[1][i]=matrix[1][i].multiply(x);
                        ans.matrix[2][i]=matrix[2][i].multiply(x);
                }
                ans.matrix[0][mRow]=matrix[0][mRow].multiply(x);
                ans.matrix[1][mRow]=matrix[1][mRow].multiply(x);
                return ans;
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v a complex vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public ComplexVector multiply(final ComplexVector v) {
                int mRow=matrix[1].length;
                if(mRow==v.dimension()) {
                        final Complex array[]=new Complex[mRow];
                        array[0]=(matrix[1][0].multiply(v.getComponent(0))).add(matrix[2][0].multiply(v.getComponent(1)));
                        mRow--;
                        for(int i=1;i<mRow;i++)
                                array[i]=(matrix[0][i].multiply(v.getComponent(i-1))).add(matrix[1][i].multiply(v.getComponent(i))).add(matrix[2][i].multiply(v.getComponent(i+1)));
                        array[mRow]=(matrix[0][mRow].multiply(v.getComponent(mRow-1))).add(matrix[1][mRow].multiply(v.getComponent(mRow)));
                        return new ComplexVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a complex matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexMatrix multiply(final ComplexMatrix m) {
                if(m instanceof ComplexTridiagonalMatrix)
                        return multiply((ComplexTridiagonalMatrix)m);
                else if(m instanceof ComplexSquareMatrix)
                        return rawMultiply((ComplexSquareMatrix)m);
                else
                        return rawMultiply(m);
        }
        private ComplexMatrix rawMultiply(final ComplexMatrix m) {
                if(matrix[1].length==m.rows()) {
                        int n,k;
                        final Complex array[][]=new Complex[matrix[1].length][m.columns()];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array[0].length;k++) {
                                        array[j][k]=getElement(j,0).multiply(m.getElement(0,k));
                                        for(n=1;n<matrix[1].length;n++)
                                                array[j][k]=array[j][k].add(getElement(j,n).multiply(m.getElement(n,k)));
                                }
                        }
                        return new ComplexMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a complex square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexSquareMatrix multiply(final ComplexSquareMatrix m) {
                if(m instanceof ComplexTridiagonalMatrix)
                        return multiply((ComplexTridiagonalMatrix)m);
                else
                        return rawMultiply(m);
        }
        private ComplexSquareMatrix rawMultiply(final ComplexSquareMatrix m) {
                if(matrix[1].length==m.rows()) {
                        int n,k;
                        final Complex array[][]=new Complex[matrix[1].length][matrix[1].length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array.length;k++) {
                                        array[j][k]=getElement(j,0).multiply(m.getElement(0,k));
                                        for(n=1;n<array.length;n++)
                                                array[j][k]=array[j][k].add(getElement(j,n).multiply(m.getElement(n,k)));
                                }
                        }
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a complex tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public ComplexSquareMatrix multiply(final ComplexTridiagonalMatrix m) {
                int mRow=matrix[1].length;
                if(mRow==m.rows()) {
                        final Complex array[][]=new Complex[mRow][mRow];
                        array[0][0]=(matrix[1][0].multiply(m.getElement(0,0))).add(matrix[2][0].multiply(m.getElement(1,0)));
                        array[0][1]=(matrix[1][0].multiply(m.getElement(0,1))).add(matrix[2][0].multiply(m.getElement(1,1)));
                        array[0][2]=matrix[2][0].multiply(m.getElement(1,2));
                        if(mRow>3) {
                                array[1][0]=(matrix[0][1].multiply(m.getElement(0,0))).add(matrix[1][1].multiply(m.getElement(1,0)));
                                array[1][1]=(matrix[0][1].multiply(m.getElement(0,1))).add(matrix[1][1].multiply(m.getElement(1,1))).add(matrix[2][1].multiply(m.getElement(2,1)));
                                array[1][2]=(matrix[1][1].multiply(m.getElement(1,2))).add(matrix[2][1].multiply(m.getElement(2,2)));
                                array[1][3]=matrix[2][1].multiply(m.getElement(2,3));
                        }
                        if(mRow==3) {
                                array[1][0]=(matrix[0][1].multiply(m.getElement(0,0))).add(matrix[1][1].multiply(m.getElement(1,0)));
                                array[1][1]=(matrix[0][1].multiply(m.getElement(0,1))).add(matrix[1][1].multiply(m.getElement(1,1))).add(matrix[2][1].multiply(m.getElement(2,1)));
                                array[1][2]=(matrix[1][1].multiply(m.getElement(1,2))).add(matrix[2][1].multiply(m.getElement(2,2)));
                        } else if(mRow>4) {
                                for(int i=2;i<mRow-2;i++) {
                                        array[i][i-2]=matrix[0][i].multiply(m.getElement(i-1,i-2));
                                        array[i][i-1]=(matrix[0][i].multiply(m.getElement(i-1,i-1))).add(matrix[1][i].multiply(m.getElement(i,i-1)));
                                        array[i][i]=(matrix[0][i].multiply(m.getElement(i-1,i))).add(matrix[1][i].multiply(m.getElement(i,i))).add(matrix[2][i].multiply(m.getElement(i+1,i)));
                                        array[i][i+1]=(matrix[1][i].multiply(m.getElement(i,i+1))).add(matrix[2][i].multiply(m.getElement(i+1,i+1)));
                                        array[i][i+2]=matrix[2][i].multiply(m.getElement(i+1,i+2));
                                }
                        }
                        if(mRow>3) {
                                array[mRow-2][mRow-4]=matrix[0][mRow-2].multiply(m.getElement(mRow-3,mRow-4));
                                array[mRow-2][mRow-3]=(matrix[0][mRow-2].multiply(m.getElement(mRow-3,mRow-3))).add(matrix[1][mRow-2].multiply(m.getElement(mRow-2,mRow-3)));
                                array[mRow-2][mRow-2]=(matrix[0][mRow-2].multiply(m.getElement(mRow-3,mRow-2))).add(matrix[1][mRow-2].multiply(m.getElement(mRow-2,mRow-2))).add(matrix[2][mRow-2].multiply(m.getElement(mRow-1,mRow-2)));
                                array[mRow-2][mRow-1]=(matrix[1][mRow-2].multiply(m.getElement(mRow-2,mRow-1))).add(matrix[2][mRow-2].multiply(m.getElement(mRow-1,mRow-1)));
                        }
                        mRow--;
                        array[mRow][mRow-2]=matrix[0][mRow].multiply(m.getElement(mRow-1,mRow-2));
                        array[mRow][mRow-1]=(matrix[0][mRow].multiply(m.getElement(mRow-1,mRow-1))).add(matrix[1][mRow].multiply(m.getElement(mRow,mRow-1)));
                        array[mRow][mRow]=(matrix[0][mRow].multiply(m.getElement(mRow-1,mRow))).add(matrix[1][mRow].multiply(m.getElement(mRow,mRow)));
                        return new ComplexSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }

// HERMITIAN ADJOINT

        /**
        * Returns the hermitian adjoint of this matrix.
        * @return a complex tridiagonal matrix
        */
        public ComplexMatrix hermitianAdjoint() {
                int mRow=matrix[1].length;
                final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(mRow);
                ans.matrix[1][0]=matrix[1][0].conjugate();
                ans.setElement(1,0,matrix[2][0].conjugate());
                mRow--;
                for(int i=1;i<mRow;i++) {
                        ans.setElement(i-1,i,matrix[0][i].conjugate());
                        ans.matrix[1][i]=matrix[1][i].conjugate();
                        ans.setElement(i+1,i,matrix[2][i].conjugate());
                }
                ans.setElement(mRow-1,mRow,matrix[0][mRow].conjugate());
                ans.matrix[1][mRow]=matrix[1][mRow].conjugate();
                return ans;
        }

// CONJUGATE

        /**
        * Returns the complex conjugate of this matrix.
        * @return a complex tridiagonal matrix
        */
        public ComplexMatrix conjugate() {
                int mRow=matrix[1].length;
                final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(mRow);
                ans.matrix[1][0]=matrix[1][0].conjugate();
                ans.matrix[2][0]=matrix[2][0].conjugate();
                mRow--;
                for(int i=1;i<mRow;i++) {
                        ans.matrix[0][i]=matrix[0][i].conjugate();
                        ans.matrix[1][i]=matrix[1][i].conjugate();
                        ans.matrix[2][i]=matrix[2][i].conjugate();
                }
                ans.matrix[0][mRow]=matrix[0][mRow].conjugate();
                ans.matrix[1][mRow]=matrix[1][mRow].conjugate();
                return ans;
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return a complex tridiagonal matrix
        */
        public Matrix transpose() {
                final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(matrix[1].length);
                System.arraycopy(matrix[0],1,ans.matrix[2],0,matrix[0].length-1);
                System.arraycopy(matrix[1],0,ans.matrix[1],0,matrix[1].length);
                System.arraycopy(matrix[2],0,ans.matrix[0],1,matrix[2].length-1);
                return ans;
        }

// LU DECOMPOSITION

        /**
        * Returns the LU decomposition of this matrix.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public ComplexSquareMatrix[] luDecompose() {
                int i,j,k;
                Complex tmp;
                final Complex array[][][]=new Complex[2][matrix[1].length][matrix[1].length];
                array[0][0][0]=Complex.ONE;
                for(i=1;i<matrix[1].length;i++)
                        array[0][i][i]=Complex.ONE;
                for(j=0;j<matrix[1].length;j++) {
                        for(i=0;i<=j;i++) {
                                tmp=getElement(i,j);
                                for(k=0;k<i;k++)
                                        tmp=tmp.subtract(array[0][i][k].multiply(array[1][k][j]));
                                array[1][i][j]=tmp;
                        }
                        for(i=j+1;i<matrix[1].length;i++) {
                                tmp=getElement(i,j);
                                for(k=0;k<j;k++)
                                        tmp=tmp.subtract(array[0][i][k].multiply(array[1][k][j]));
                                array[0][i][j]=tmp.divide(array[1][j][j]);
                        }
                }
                final ComplexSquareMatrix lu[]=new ComplexSquareMatrix[2];
                lu[0]=new ComplexSquareMatrix(array[0]);
                lu[1]=new ComplexSquareMatrix(array[1]);
                return lu;
        }

// MAP ELEMENTS

        /**
        * Applies a function on all the matrix elements.
        * @param f a user-defined function
        * @return a complex tridiagonal matrix
        */
        public ComplexMatrix mapElements(final Mapping f) {
                int mRow=matrix[1].length;
                final ComplexTridiagonalMatrix ans=new ComplexTridiagonalMatrix(mRow);
                ans.matrix[1][0]=f.map(matrix[1][0]);
                ans.matrix[2][0]=f.map(matrix[2][0]);
                mRow--;
                for(int i=1;i<mRow;i++) {
                        ans.matrix[0][i]=f.map(matrix[0][i]);
                        ans.matrix[1][i]=f.map(matrix[1][i]);
                        ans.matrix[2][i]=f.map(matrix[2][i]);
                }
                ans.matrix[0][mRow]=f.map(matrix[0][mRow]);
                ans.matrix[1][mRow]=f.map(matrix[1][mRow]);
                return ans;
        }
}

