package JSci.maths;

import JSci.maths.groups.AbelianGroupMember;

/**
* The ComplexVector class encapsulates vectors containing complex numbers.
* @version 2.1
* @author Mark Hale
*/
public class ComplexVector extends MathVector {
        /**
        * Storage format identifier.
        */
        protected final static int ARRAY_1D=1;
        protected final static int storageFormat=ARRAY_1D;
        /**
        * Array containing the components of the vector.
        */
        protected Complex vector[];
        /**
        * Constructs an empty vector.
        * @param dim the dimension of the vector.
        */
        public ComplexVector(final int dim) {
                vector=new Complex[dim];
        }
        /**
        * Constructs a vector by wrapping an array.
        * @param array an assigned value
        */
        public ComplexVector(final Complex array[]) {
                vector=array;
        }
        /**
        * Compares two complex vectors for equality.
        * @param a a complex vector
        */
        public boolean equals(Object a) {
                if(a!=null && (a instanceof ComplexVector) && vector.length==((ComplexVector)a).dimension()) {
                        final ComplexVector cv=(ComplexVector)a;
                        for(int i=0;i<vector.length;i++) {
                                if(!vector[i].equals(cv.getComponent(i)))
                                        return false;
                        }
                        return true;
                } else
                        return false;
        }
        /**
        * Returns a comma delimited string representing the value of this vector.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer(vector.length);
                int i;
                for(i=0;i<vector.length-1;i++) {
                        buf.append(vector[i].toString());
                        buf.append(',');
                }
                buf.append(vector[i].toString());
                return buf.toString();
        }
        /**
        * Returns a hashcode for this vector.
        */
        public int hashCode() {
                return (int)Math.exp(norm());
        }
        /**
        * Returns the real part of this complex vector.
        */
        public DoubleVector real() {
                final double array[]=new double[vector.length];
                for(int i=0;i<vector.length;i++)
                        array[i]=vector[i].real();
                return new DoubleVector(array);
        }
        /**
        * Returns the imaginary part of this complex vector.
        */
        public DoubleVector imag() {
                final double array[]=new double[vector.length];
                for(int i=0;i<vector.length;i++)
                        array[i]=vector[i].imag();
                return new DoubleVector(array);
        }
        /**
        * Returns a component of this vector.
        * @param n index of the vector component
        * @exception VectorDimensionException If attempting to access an invalid component.
        */
        public Complex getComponent(final int n) {
                if(n>=0 && n<vector.length)
                        return vector[n];
                else
                        throw new VectorDimensionException("Invalid component.");
        }
        /**
        * Sets the value of a component of this vector.
        * @param n index of the vector component
        * @param z a complex number
        * @exception VectorDimensionException If attempting to access an invalid component.
        */
        public void setComponent(final int n, final Complex z) {
                if(n>=0 && n<vector.length)
                        vector[n]=z;
                else
                        throw new VectorDimensionException("Invalid component.");
        }
        /**
        * Returns the vector's dimension.
        */
        public int dimension() {
                return vector.length;
        }
        /**
        * Returns the l2-norm (magnitude).
        */
        public double norm() {
                double answer=vector[0].real()*vector[0].real()+vector[0].imag()*vector[0].imag();
                for(int i=1;i<vector.length;i++)
                        answer+=vector[i].real()*vector[i].real()+vector[i].imag()*vector[i].imag();
                return Math.sqrt(answer);
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public double infNorm() {
                double infNorm=vector[0].mod();
                for(int i=1;i<vector.length;i++) {
                        if(vector[i].mod()>infNorm)
                                infNorm=vector[i].mod();
                }
                return infNorm;
        }

//============
// OPERATIONS
//============

        /**
        * Returns the negative of this vector.
        */
        public AbelianGroupMember negate() {
                final Complex array[]=new Complex[vector.length];
                array[0]=(Complex)vector[0].negate();
                for(int i=1;i<array.length;i++)
                        array[i]=(Complex)vector[i].negate();
                return new ComplexVector(array);
        }

// COMPLEX CONJUGATE

        /**
        * Returns the complex conjugate of this vector.
        */
        public ComplexVector conjugate() {
                final Complex array[]=new Complex[vector.length];
                array[0]=vector[0].conjugate();
                for(int i=1;i<array.length;i++)
                        array[i]=vector[i].conjugate();
                return new ComplexVector(array);
        }

// ADDITION

        /**
        * Returns the addition of this vector and another.
        */
        public AbelianGroupMember add(final AbelianGroupMember v) {
                if(v instanceof ComplexVector)
                        return add((ComplexVector)v);
                else if(v instanceof DoubleVector)
                        return add((DoubleVector)v);
                else if(v instanceof IntegerVector)
                        return add((IntegerVector)v);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the addition of this vector and another.
        * @param v a complex vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public ComplexVector add(final ComplexVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawAdd(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        final Complex array[]=new Complex[vector.length];
                                        array[0]=vector[0].add(v.getComponent(0));
                                        for(int i=1;i<array.length;i++)
                                                array[i]=vector[i].add(v.getComponent(i));
                                        return new ComplexVector(array);
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private ComplexVector rawAdd(final ComplexVector v) {
                if(vector.length==v.vector.length) {
                        final Complex array[]=new Complex[vector.length];
                        array[0]=vector[0].add(v.vector[0]);
                        for(int i=1;i<array.length;i++)
                                array[i]=vector[i].add(v.vector[i]);
                        return new ComplexVector(array);
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }
        /**
        * Returns the addition of this vector and another.
        * @param v a double vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public ComplexVector add(final DoubleVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawAdd(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        final Complex array[]=new Complex[vector.length];
                                        array[0]=vector[0].addReal(v.getComponent(0));
                                        for(int i=1;i<array.length;i++)
                                                array[i]=vector[i].addReal(v.getComponent(i));
                                        return new ComplexVector(array);
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private ComplexVector rawAdd(final DoubleVector v) {
                if(vector.length==v.vector.length) {
                        final Complex array[]=new Complex[vector.length];
                        array[0]=vector[0].addReal(v.vector[0]);
                        for(int i=1;i<array.length;i++)
                                array[i]=vector[i].addReal(v.vector[i]);
                        return new ComplexVector(array);
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }
        /**
        * Returns the addition of this vector and another.
        * @param v an integer vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public ComplexVector add(final IntegerVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawAdd(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        final Complex array[]=new Complex[vector.length];
                                        array[0]=vector[0].addReal(v.getComponent(0));
                                        for(int i=1;i<array.length;i++)
                                                array[i]=vector[i].addReal(v.getComponent(i));
                                        return new ComplexVector(array);
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private ComplexVector rawAdd(final IntegerVector v) {
                if(vector.length==v.vector.length) {
                        final Complex array[]=new Complex[vector.length];
                        array[0]=vector[0].addReal(v.vector[0]);
                        for(int i=1;i<array.length;i++)
                                array[i]=vector[i].addReal(v.vector[i]);
                        return new ComplexVector(array);
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this vector by another.
        */
        public AbelianGroupMember subtract(final AbelianGroupMember v) {
                if(v instanceof ComplexVector)
                        return subtract((ComplexVector)v);
                else if(v instanceof DoubleVector)
                        return subtract((DoubleVector)v);
                else if(v instanceof IntegerVector)
                        return subtract((IntegerVector)v);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the subtraction of this vector by another.
        * @param v a complex vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public ComplexVector subtract(final ComplexVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawSubtract(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        final Complex array[]=new Complex[vector.length];
                                        array[0]=vector[0].subtract(v.getComponent(0));
                                        for(int i=1;i<array.length;i++)
                                                array[i]=vector[i].subtract(v.getComponent(i));
                                        return new ComplexVector(array);
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private ComplexVector rawSubtract(final ComplexVector v) {
                if(vector.length==v.vector.length) {
                        final Complex array[]=new Complex[vector.length];
                        array[0]=vector[0].subtract(v.vector[0]);
                        for(int i=1;i<array.length;i++)
                                array[i]=vector[i].subtract(v.vector[i]);
                        return new ComplexVector(array);
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }
        /**
        * Returns the subtraction of this vector by another.
        * @param v a double vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public ComplexVector subtract(final DoubleVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawSubtract(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        final Complex array[]=new Complex[vector.length];
                                        array[0]=vector[0].subtractReal(v.getComponent(0));
                                        for(int i=1;i<array.length;i++)
                                                array[i]=vector[i].subtractReal(v.getComponent(i));
                                        return new ComplexVector(array);
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private ComplexVector rawSubtract(final DoubleVector v) {
                if(vector.length==v.vector.length) {
                        final Complex array[]=new Complex[vector.length];
                        array[0]=vector[0].subtractReal(v.vector[0]);
                        for(int i=1;i<array.length;i++)
                                array[i]=vector[i].subtractReal(v.vector[i]);
                        return new ComplexVector(array);
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }
        /**
        * Returns the subtraction of this vector by another.
        * @param v an integer vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public ComplexVector subtract(final IntegerVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawSubtract(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        final Complex array[]=new Complex[vector.length];
                                        array[0]=vector[0].subtractReal(v.getComponent(0));
                                        for(int i=1;i<array.length;i++)
                                                array[i]=vector[i].subtractReal(v.getComponent(i));
                                        return new ComplexVector(array);
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private ComplexVector rawSubtract(final IntegerVector v) {
                if(vector.length==v.vector.length) {
                        final Complex array[]=new Complex[vector.length];
                        array[0]=vector[0].subtractReal(v.vector[0]);
                        for(int i=1;i<array.length;i++)
                                array[i]=vector[i].subtractReal(v.vector[i]);
                        return new ComplexVector(array);
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this vector by a scalar.
        */
        public ModuleMember scalarMultiply(RingMember x) {
                if(x instanceof Complex)
                        return scalarMultiply((Complex)x);
                else if(x instanceof MathDouble)
                        return scalarMultiply(((MathDouble)x).value());
                else if(x instanceof MathInteger)
                        return scalarMultiply(((MathInteger)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the multiplication of this vector by a scalar.
        * @param z a complex number
        */
        public ComplexVector scalarMultiply(final Complex z) {
                final Complex array[]=new Complex[vector.length];
                array[0]=vector[0].multiply(z);
                for(int i=1;i<array.length;i++)
                        array[i]=vector[i].multiply(z);
                return new ComplexVector(array);
        }
        /**
        * Returns the multiplication of this vector by a scalar.
        * @param x a double
        */
        public ComplexVector scalarMultiply(final double x) {
                final Complex array[]=new Complex[vector.length];
                array[0]=vector[0].multiply(x);
                for(int i=1;i<array.length;i++)
                        array[i]=vector[i].multiply(x);
                return new ComplexVector(array);
        }

// SCALAR DIVISION

        /**
        * Returns the division of this vector by a scalar.
        * @param z a complex number
        * @exception ArithmeticException If divide by zero.
        */
        public ComplexVector scalarDivide(final Complex z) {
                final Complex array[]=new Complex[vector.length];
                array[0]=vector[0].divide(z);
                for(int i=1;i<array.length;i++)
                        array[i]=vector[i].divide(z);
                return new ComplexVector(array);
        }
        /**
        * Returns the division of this vector by a scalar.
        * @param x a double
        * @exception ArithmeticException If divide by zero.
        */
        public ComplexVector scalarDivide(final double x) {
                final Complex array[]=new Complex[vector.length];
                array[0]=vector[0].divide(x);
                for(int i=1;i<array.length;i++)
                        array[i]=vector[i].divide(x);
                return new ComplexVector(array);
        }

// SCALAR PRODUCT

        /**
        * Returns the scalar product of this vector and another.
        * @param v a complex vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public Complex scalarProduct(final ComplexVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawScalarProduct(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        double compRe=v.getComponent(0).real();
                                        double compIm=v.getComponent(0).imag();
                                        double real=vector[0].real()*compRe+vector[0].imag()*compIm;
                                        double imag=vector[0].imag()*compRe-vector[0].real()*compIm;
                                        for(int i=1;i<vector.length;i++) {
                                                compRe=v.getComponent(i).real();
                                                compIm=v.getComponent(i).imag();
                                                real+=vector[i].real()*compRe+vector[i].imag()*compIm;
                                                imag+=vector[i].imag()*compRe-vector[i].real()*compIm;
                                        }
                                        return new Complex(real,imag);
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private Complex rawScalarProduct(final ComplexVector v) {
                if(vector.length==v.vector.length) {
                        double real=vector[0].real()*v.vector[0].real()+vector[0].imag()*v.vector[0].imag();
                        double imag=vector[0].imag()*v.vector[0].real()-vector[0].real()*v.vector[0].imag();
                        for(int i=1;i<vector.length;i++) {
                                real+=vector[i].real()*v.vector[i].real()+vector[i].imag()*v.vector[i].imag();
                                imag+=vector[i].imag()*v.vector[i].real()-vector[i].real()*v.vector[i].imag();
                        }
                        return new Complex(real,imag);
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }

// MAP COMPONENTS

        /**
        * Applies a function on all the vector components.
        * @param f a user-defined function
        * @return a complex vector
        */
        public ComplexVector mapComponents(final Mapping f) {
                final Complex array[]=new Complex[vector.length];
                array[0]=f.map(vector[0]);
                for(int i=1;i<vector.length;i++)
                        array[i]=f.map(vector[i]);
                return new ComplexVector(array);
        }
}

