package JSci.maths;

import JSci.GlobalSettings;

/**
* The DoubleDiagonalMatrix class encapsulates diagonal matrices containing doubles.
* Uses compressed diagonal storage.
* @version 2.1
* @author Mark Hale
*/
public final class DoubleDiagonalMatrix extends DoubleTridiagonalMatrix {
        /**
        * Storage format identifier.
        */
        protected final static int DIAGONAL=4;
        protected final static int storageFormat=DIAGONAL;
        /**
        * Constructs a matrix.
        */
        protected DoubleDiagonalMatrix() {
                super();
        }
        /**
        * Constructs an empty matrix.
        * @param size the number of rows/columns
        */
        public DoubleDiagonalMatrix(final int size) {
                this();
                matrix=new double[1][size];
        }
        /**
        * Constructs a matrix from an array.
        * @param array an assigned value
        * @exception MatrixDimensionException If the array is not square.
        */
        public DoubleDiagonalMatrix(final double array[][]) {
                this(array.length);
                if(array.length==array[0].length) {
                        for(int i=0;i<matrix[0].length;i++)
                                matrix[0][i]=array[i][i];
                } else {
                        matrix=null;
                        throw new MatrixDimensionException("Array must be square.");
                }
        }
        /**
        * Constructs a matrix by wrapping an array containing the diagonal elements.
        * @param array an assigned value
        */
        public DoubleDiagonalMatrix(final double array[]) {
                this();
                matrix=new double[1][];
                matrix[0]=array;
        }
        /**
        * Creates an identity matrix.
        * @param size the number of rows/columns
        */
        public static DoubleDiagonalMatrix identity(final int size) {
                double array[]=new double[size];
                for(int i=0;i<size;i++)
                        array[i]=1.0;
                return new DoubleDiagonalMatrix(array);
        }
        /**
        * Compares two double diagonal matrices for equality.
        * @param m a double diagonal matrix
        */
        public boolean equals(Object m) {
                if(m!=null && (m instanceof DoubleDiagonalMatrix) &&
                matrix[0].length==((DoubleDiagonalMatrix)m).rows()) {
                        final DoubleDiagonalMatrix ddm=(DoubleDiagonalMatrix)m;
                        for(int i=0;i<matrix[0].length;i++) {
                                if(Math.abs(matrix[0][i]-ddm.getElement(i,i))>GlobalSettings.ZERO_TOL)
                                        return false;
                        }
                        return true;
                } else
                        return false;
        }
        /**
        * Converts this matrix to an integer matrix.
        * @return an integer diagonal matrix
        */
        public IntegerMatrix toIntegerMatrix() {
                final int array[]=new int[matrix[0].length];
                for(int i=0;i<matrix[0].length;i++)
                        array[i]=Math.round((float)matrix[0][i]);
                return new IntegerDiagonalMatrix(array);
        }
        /**
        * Converts this matrix to a complex matrix.
        * @return a complex diagonal matrix
        */
        public ComplexMatrix toComplexMatrix() {
                final Complex array[]=new Complex[matrix[0].length];
                for(int i=0;i<matrix[0].length;i++)
                        array[i]=new Complex(matrix[0][i],0.0);
                return new ComplexDiagonalMatrix(array);
        }
        /**
        * Returns an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public double getElement(final int i, final int j) {
                if(i>=0 && i<matrix[0].length && j>=0 && j<matrix[0].length) {
                        if(i==j)
                                return matrix[0][i];
                        else
                                return 0.0;
                } else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Sets the value of an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @param x a number
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public void setElement(final int i, final int j, final double x) {
                if(i>=0 && i<matrix[0].length && j>=0 && j<matrix[0].length && i==j)
                        matrix[0][i]=x;
                else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Returns true if this matrix is symmetric.
        */
        public boolean isSymmetric() {
                return true;
        }
        /**
        * Returns true if this matrix is unitary.
        */
        public boolean isUnitary() {
                return this.multiply(this).equals(identity(matrix[0].length));
        }
        /**
        * Returns the determinant.
        */
        public double det() {
                double det=matrix[0][0];
                for(int i=1;i<matrix[0].length;i++)
                        det*=matrix[0][i];
                return det;
        }
        /**
        * Returns the trace.
        */
        public double trace() {
                double tr=matrix[0][0];
                for(int i=1;i<matrix[0].length;i++)
                        tr+=matrix[0][i];
                return tr;
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public double infNorm() {
                double result=Math.abs(matrix[0][0]);
                double tmpResult;
                for(int i=1;i<matrix[0].length;i++) {
                        tmpResult=Math.abs(matrix[0][i]);
                        if(tmpResult>result)
                                result=tmpResult;
                }
                return result;
        }
        /**
        * Returns the Frobenius norm.
        * @author Taber Smith
        */
        public double frobeniusNorm() {
                double result=matrix[0][0]*matrix[0][0];
                for(int i=1;i<matrix[0].length;i++)
                        result+=matrix[0][i]*matrix[0][i];
                return Math.sqrt(result);
        }
        /**
        * Returns the number of rows.
        */
        public int rows() {
                return matrix[0].length;
        }
        /**
        * Returns the number of columns.
        */
        public int columns() {
                return matrix[0].length;
        }

//============
// OPERATIONS
//============

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix add(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        case TRIDIAGONAL: return rawAdd((DoubleTridiagonalMatrix)m);
                        case DIAGONAL: return rawAdd((DoubleDiagonalMatrix)m);
                        default: 
                                if(matrix[0].length==m.rows() && matrix[0].length==m.columns()) {
                                        final double array[][]=new double[matrix[0].length][matrix[0].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                       array[i][j]=m.getElement(i,j);
                                        }
                                        for(int i=0;i<array.length;i++)
                                                array[i][i]+=matrix[0][i];
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleSquareMatrix rawAdd(final DoubleMatrix m) {
                if(matrix[0].length==m.matrix.length && matrix[0].length==m.matrix[0].length) {
                        final double array[][]=new double[matrix[0].length][matrix[0].length];
                        for(int i=0;i<array.length;i++)
                                System.arraycopy(m.matrix[i],0,array[i],0,array.length);
                        for(int i=0;i<array.length;i++)
                                array[i][i]+=matrix[0][i];
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix add(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        case TRIDIAGONAL: return rawAdd((DoubleTridiagonalMatrix)m);
                        case DIAGONAL: return rawAdd((DoubleDiagonalMatrix)m);
                        default: 
                                if(matrix[0].length==m.rows()) {
                                        final double array[][]=new double[matrix[0].length][matrix[0].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                       array[i][j]=m.getElement(i,j);
                                        }
                                        for(int i=0;i<array.length;i++)
                                                array[i][i]+=matrix[0][i];
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a double tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleTridiagonalMatrix add(final DoubleTridiagonalMatrix m) {
                switch(m.storageFormat) {
                        case TRIDIAGONAL: return rawAdd(m);
                        case DIAGONAL: return rawAdd((DoubleDiagonalMatrix)m);
                        default: 
                                int mRow=matrix[0].length;
                                if(mRow==m.rows()) {
                                        final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(mRow);
                                        ans.matrix[1][0]=matrix[0][0]+m.getElement(0,0);
                                        ans.matrix[2][0]=m.getElement(0,1);
                                        mRow--;
                                        for(int i=1;i<mRow;i++) {
                                                ans.matrix[0][i]=m.getElement(i,i-1);
                                                ans.matrix[1][i]=matrix[0][i]+m.getElement(i,i);
                                                ans.matrix[2][i]=m.getElement(i,i+1);
                                        }
                                        ans.matrix[0][mRow]=m.getElement(mRow,mRow-1);
                                        ans.matrix[1][mRow]=matrix[0][mRow]+m.getElement(mRow,mRow);
                                        return ans;
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleTridiagonalMatrix rawAdd(final DoubleTridiagonalMatrix m) {
                if(matrix[0].length==m.matrix[1].length) {
                        final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(matrix[0].length);
                        System.arraycopy(m.matrix[0],0,ans.matrix[0],0,matrix[0].length);
                        System.arraycopy(m.matrix[2],0,ans.matrix[2],0,matrix[2].length);
                        ans.matrix[1][0]=matrix[0][0]+m.matrix[1][0];
                        for(int i=1;i<matrix[0].length;i++)
                                ans.matrix[1][i]=matrix[0][i]+m.matrix[1][i];
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a double diagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleDiagonalMatrix add(final DoubleDiagonalMatrix m) {
                if(matrix[0].length==m.matrix[0].length) {
                        final double array[]=new double[matrix[0].length];
                        array[0]=matrix[0][0]+m.matrix[0][0];
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i]+m.matrix[0][i];
                        return new DoubleDiagonalMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix subtract(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        case TRIDIAGONAL: return rawSubtract((DoubleTridiagonalMatrix)m);
                        case DIAGONAL: return rawSubtract((DoubleDiagonalMatrix)m);
                        default: 
                                if(matrix[0].length==m.rows() && matrix[0].length==m.columns()) {
                                        final double array[][]=new double[matrix[0].length][matrix[0].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=-m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                       array[i][j]=-m.getElement(i,j);
                                        }
                                        for(int i=0;i<array.length;i++)
                                                array[i][i]+=matrix[0][i];
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleSquareMatrix rawSubtract(final DoubleMatrix m) {
                if(matrix[0].length==m.matrix.length && matrix[0].length==m.matrix[0].length) {
                        final double array[][]=new double[matrix[0].length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=-m.matrix[i][0];
                                for(j=1;j<array.length;j++)
                                        array[i][j]=-m.matrix[i][j];
                        }
                        for(int i=0;i<array.length;i++)
                                array[i][i]+=matrix[0][i];
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix and another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix subtract(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        case TRIDIAGONAL: return rawSubtract((DoubleTridiagonalMatrix)m);
                        case DIAGONAL: return rawSubtract((DoubleDiagonalMatrix)m);
                        default: 
                                if(matrix[0].length==m.rows()) {
                                        final double array[][]=new double[matrix[0].length][matrix[0].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=-m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                       array[i][j]=-m.getElement(i,j);
                                        }
                                        for(int i=0;i<array.length;i++)
                                                array[i][i]+=matrix[0][i];
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        /**
        * Returns the subtraction of this matrix and another.
        * @param m a double tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleTridiagonalMatrix subtract(final DoubleTridiagonalMatrix m) {
                switch(m.storageFormat) {
                        case TRIDIAGONAL: return rawSubtract(m);
                        case DIAGONAL: return rawSubtract((DoubleDiagonalMatrix)m);
                        default: 
                                int mRow=matrix[0].length;
                                if(mRow==m.rows()) {
                                        final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(mRow);
                                        ans.matrix[1][0]=matrix[0][0]-m.getElement(0,0);
                                        ans.matrix[2][0]=-m.getElement(0,1);
                                        mRow--;
                                        for(int i=1;i<mRow;i++) {
                                                ans.matrix[0][i]=-m.getElement(i,i-1);
                                                ans.matrix[1][i]=matrix[0][i]-m.getElement(i,i);
                                                ans.matrix[2][i]=-m.getElement(i,i+1);
                                        }
                                        ans.matrix[0][mRow]=-m.getElement(mRow,mRow-1);
                                        ans.matrix[1][mRow]=matrix[0][mRow]-m.getElement(mRow,mRow);
                                        return ans;
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleTridiagonalMatrix rawSubtract(final DoubleTridiagonalMatrix m) {
                int mRow=matrix[0].length;
                if(mRow==m.matrix[1].length) {
                        final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(mRow);
                        ans.matrix[1][0]=matrix[0][0]-m.matrix[1][0];
                        ans.matrix[2][0]=-m.matrix[2][0];
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.matrix[0][i]=-m.matrix[0][i];
                                ans.matrix[1][i]=matrix[0][i]-m.matrix[1][i];
                                ans.matrix[2][i]=-m.matrix[2][i];
                        }
                        ans.matrix[0][mRow]=-m.matrix[0][mRow];
                        ans.matrix[1][mRow]=matrix[0][mRow]-m.matrix[1][mRow];
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a double diagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleDiagonalMatrix subtract(final DoubleDiagonalMatrix m) {
                if(matrix[0].length==m.matrix[0].length) {
                        final double array[]=new double[matrix[0].length];
                        array[0]=matrix[0][0]-m.matrix[0][0];
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i]-m.matrix[0][i];
                        return new DoubleDiagonalMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SCALAR MULTIPLY

        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x a double
        * @return a double diagonal matrix
        */
        public DoubleMatrix scalarMultiply(final double x) {
                final double array[]=new double[matrix[0].length];
                array[0]=x*matrix[0][0];
                for(int i=1;i<array.length;i++)
                        array[i]=x*matrix[0][i];
                return new DoubleDiagonalMatrix(array);
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v a double vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public DoubleVector multiply(final DoubleVector v) {
                if(matrix[0].length==v.dimension()) {
                        final double array[]=new double[matrix[0].length];
                        array[0]=matrix[0][0]*v.getComponent(0);
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i]*v.getComponent(i);
                        return new DoubleVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix multiply(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        case TRIDIAGONAL: return rawMultiply((DoubleTridiagonalMatrix)m);
                        case DIAGONAL: return rawMultiply((DoubleDiagonalMatrix)m);
                        default: 
                                if(matrix[0].length==m.rows()) {
                                        final double array[][]=new double[matrix[0].length][m.columns()];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[0][i]*m.getElement(i,0);
                                                for(j=1;j<array[0].length;j++)
                                                        array[i][j]=matrix[0][i]*m.getElement(i,j);
                                        }
                                        return new DoubleMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleMatrix rawMultiply(final DoubleMatrix m) {
                if(matrix[0].length==m.matrix.length) {
                        final double array[][]=new double[matrix[0].length][m.matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[0][i]*m.matrix[i][0];
                                for(j=1;j<array[0].length;j++)
                                        array[i][j]=matrix[0][i]*m.matrix[i][j];
                        }
                        return new DoubleMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix multiply(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        case TRIDIAGONAL: return rawMultiply((DoubleTridiagonalMatrix)m);
                        case DIAGONAL: return rawMultiply((DoubleDiagonalMatrix)m);
                        default: 
                                if(matrix[0].length==m.rows()) {
                                        final double array[][]=new double[matrix[0].length][matrix[0].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=matrix[0][i]*m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=matrix[0][i]*m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleSquareMatrix rawMultiply(final DoubleSquareMatrix m) {
                if(matrix[0].length==m.matrix.length) {
                        final double array[][]=new double[matrix[0].length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[0][i]*m.matrix[i][0];
                                for(j=1;j<array.length;j++)
                                        array[i][j]=matrix[0][i]*m.matrix[i][j];
                        }
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix multiply(final DoubleTridiagonalMatrix m) {
                switch(m.storageFormat) {
                        case TRIDIAGONAL: return rawMultiply((DoubleTridiagonalMatrix)m);
                        case DIAGONAL: return rawMultiply((DoubleDiagonalMatrix)m);
                        default: 
                                int mRow=matrix[0].length;
                                if(mRow==m.rows()) {
                                        final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(mRow);
                                        ans.matrix[1][0]=matrix[0][0]*m.getElement(0,0);
                                        ans.matrix[2][0]=matrix[0][0]*m.getElement(0,1);
                                        mRow--;
                                        for(int i=1;i<mRow;i++) {
                                                ans.matrix[0][i]=matrix[0][i]*m.getElement(i,i-1);
                                                ans.matrix[1][i]=matrix[0][i]*m.getElement(i,i);
                                                ans.matrix[2][i]=matrix[0][i]*m.getElement(i,i+1);
                                        }
                                        ans.matrix[0][mRow]=matrix[0][mRow]*m.getElement(mRow,mRow-1);
                                        ans.matrix[1][mRow]=matrix[0][mRow]*m.getElement(mRow,mRow);
                                        return ans;
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                        }
        }
        private DoubleSquareMatrix rawMultiply(final DoubleTridiagonalMatrix m) {
                int mRow=matrix[0].length;
                if(mRow==m.matrix[1].length) {
                        final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(mRow);
                        ans.matrix[1][0]=matrix[0][0]*m.matrix[1][0];
                        ans.matrix[2][0]=matrix[0][0]*m.matrix[2][0];
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.matrix[0][i]=matrix[0][i]*m.matrix[0][i];
                                ans.matrix[1][i]=matrix[0][i]*m.matrix[1][i];
                                ans.matrix[2][i]=matrix[0][i]*m.matrix[2][i];
                        }
                        ans.matrix[0][mRow]=matrix[0][mRow]*m.matrix[0][mRow];
                        ans.matrix[1][mRow]=matrix[0][mRow]*m.matrix[1][mRow];
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double diagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleDiagonalMatrix multiply(final DoubleDiagonalMatrix m) {
                switch(m.storageFormat) {
                        case DIAGONAL: return rawMultiply((DoubleDiagonalMatrix)m);
                        default: 
                                if(matrix[0].length==m.rows()) {
                                        final double array[]=new double[matrix[0].length];
                                        array[0]=matrix[0][0]*m.getElement(0,0);
                                        for(int i=1;i<array.length;i++)
                                                array[i]=matrix[0][i]*m.getElement(i,i);
                                        return new DoubleDiagonalMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleDiagonalMatrix rawMultiply(final DoubleDiagonalMatrix m) {
                if(matrix[0].length==m.matrix[0].length) {
                        final double array[]=new double[matrix[0].length];
                        array[0]=matrix[0][0]*m.matrix[0][0];
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i]*m.matrix[0][i];
                        return new DoubleDiagonalMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// INVERSE

        /**
        * Returns the inverse of this matrix.
        * @return a double diagonal matrix
        */
        public DoubleSquareMatrix inverse() {
                final double array[]=new double[matrix[0].length];
                array[0]=1.0/matrix[0][0];
                for(int i=1;i<array.length;i++)
                        array[i]=1.0/matrix[0][i];
                return new DoubleDiagonalMatrix(array);
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return a double diagonal matrix
        */
        public Matrix transpose() {
                return this;
        }

// LU DECOMPOSITION

        /**
        * Returns the LU decomposition of this matrix.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] luDecompose() {
                final DoubleDiagonalMatrix lu[]=new DoubleDiagonalMatrix[2];
                lu[0]=identity(matrix[0].length);
                lu[1]=this;
                return lu;
        }

// CHOLESKY DECOMPOSITION

        /**
        * Returns the Cholesky decomposition of this matrix.
        * Matrix must be symmetric and positive definite.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] choleskyDecompose() {
                final DoubleDiagonalMatrix lu[]=new DoubleDiagonalMatrix[2];
                final double array[]=new double[matrix[0].length];
                array[0]=Math.sqrt(matrix[0][0]);
                for(int i=1;i<array.length;i++)
                        array[i]=Math.sqrt(matrix[0][i]);
                lu[0]=new DoubleDiagonalMatrix(array);
                lu[1]=lu[0];
                return lu;
        }

// MAP ELEMENTS

        /**
        * Applies a function on all the matrix elements.
        * @param f a user-defined function
        * @return a double diagonal matrix
        */
        public DoubleMatrix mapElements(final Mapping f) {
                final double array[]=new double[matrix[0].length];
                array[0]=f.map(matrix[0][0]);
                for(int i=1;i<array.length;i++)
                        array[i]=f.map(matrix[0][i]);
                return new DoubleDiagonalMatrix(array);
        }
}

