package JSci.maths;

import JSci.GlobalSettings;

/**
* The DoubleSparseMatrix class provides an object for encapsulating sparse matrices.
* Uses compressed row storage.
* @version 1.0
* @author Mark Hale
*/
public final class DoubleSparseMatrix extends DoubleSquareMatrix {
        /**
        * Storage format identifier.
        */
        protected final static int SPARSE=2;
        protected final static int storageFormat=SPARSE;
        /**
        * Sparse indexing data.
        */
        private int colPos[];
        private int rows[];
        private int N;
        /**
        * Constructs a matrix.
        */
        protected DoubleSparseMatrix() {
                super();
        }
        /**
        * Constructs an empty matrix.
        * @param size the number of rows/columns
        */
        public DoubleSparseMatrix(final int size) {
                this();
                N=size;
                matrix=new double[1][0];
                colPos=new int[0];
                rows=new int[N+1];
        }
        /**
        * Finalize.
        * @exception Throwable Any that occur.
        */
        protected void finalize() throws Throwable {
                colPos=null;
                rows=null;
                super.finalize();
        }
        /**
        * Compares two double sparse matrices for equality.
        * @param m a double matrix
        */
        public boolean equals(Object m) {
                if(m!=null && (m instanceof DoubleSparseMatrix) &&
                N==((DoubleSparseMatrix)m).N) {
                        final DoubleSparseMatrix dsm=(DoubleSparseMatrix)m;
                        if(colPos.length!=dsm.colPos.length)
                                return false;
                        for(int i=1;i<rows.length;i++) {
                                if(rows[i]!=dsm.rows[i])
                                        return false;
                        }
                        for(int i=1;i<colPos.length;i++) {
                                if(colPos[i]!=dsm.colPos[i])
                                        return false;
                                if(Math.abs(matrix[0][i]-dsm.matrix[0][i])>GlobalSettings.ZERO_TOL)
                                        return false;
                        }
                        return true;
                } else
                        return false;
        }
        /**
        * Returns a string representing this matrix.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer(N*N);
                for(int j,i=0;i<N;i++) {
                        for(j=0;j<N;j++) {
                                buf.append(getElement(i,j));
                                buf.append(' ');
                        }
                        buf.append('\n');
                }
                return buf.toString();
        }
        /**
        * Converts this matrix to an integer matrix.
        * @return an integer square matrix
        */
        public IntegerMatrix toIntegerMatrix() {
                final int ans[][]=new int[N][N];
                for(int j,i=0;i<N;i++) {
                        for(j=0;j<N;j++)
                                ans[i][j]=Math.round((float)getElement(i,j));
                }
                return new IntegerSquareMatrix(ans);
        }
        /**
        * Converts this matrix to a complex matrix.
        * @return a complex square matrix
        */
        public ComplexMatrix toComplexMatrix() {
                final Complex ans[][]=new Complex[N][N];
                for(int j,i=0;i<N;i++) {
                        for(j=0;j<N;j++)
                                ans[i][j]=new Complex(getElement(i,j),0.0);
                }
                return new ComplexSquareMatrix(ans);
        }
        /**
        * Returns an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public double getElement(final int i, final int j) {
                if(i>=0 && i<N && j>=0 && j<N) {
                        for(int k=rows[i];k<rows[i+1];k++) {
                                if(colPos[k]==j)
                                        return matrix[0][k];
                        }
                        return 0.0;
                } else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Sets the value of an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @param x a number
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public void setElement(final int i, final int j, final double x) {
                if(i>=0 && i<N && j>=0 && j<N) {
                        if(Math.abs(x)<=GlobalSettings.ZERO_TOL)
                                return;
                        int k;
                        for(k=rows[i];k<rows[i+1];k++) {
                                if(colPos[k]==j) {
                                        matrix[0][k]=x;
                                        return;
                                }
                        }
                        final double oldMatrix[]=matrix[0];
                        final int oldColPos[]=colPos;
                        matrix[0]=new double[oldMatrix.length+1];
                        colPos=new int[oldColPos.length+1];
                        System.arraycopy(oldMatrix,0,matrix[0],0,rows[i]);
                        System.arraycopy(oldColPos,0,colPos,0,rows[i]);
                        for(k=rows[i];k<rows[i+1] && oldColPos[k]<j;k++) {
                                matrix[0][k]=oldMatrix[k];
                                colPos[k]=oldColPos[k];
                        }
                        matrix[0][k]=x;
                        colPos[k]=j;
                        System.arraycopy(oldMatrix,k,matrix[0],k+1,oldMatrix.length-k);
                        System.arraycopy(oldColPos,k,colPos,k+1,oldColPos.length-k);
                        for(k=i+1;k<rows.length;k++)
                                rows[k]++;
                } else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Returns the determinant.
        */
        public double det() {
                final DoubleSquareMatrix lu[]=this.luDecompose();
                double det=lu[1].matrix[0][0];
                for(int i=1;i<N;i++)
                        det*=lu[1].matrix[i][i];
                return det;
        }
        /**
        * Returns the trace.
        * @author Taber Smith
        */
        public double trace() {
                double result=getElement(0,0);
                for(int i=1;i<N;i++)
                        result+=getElement(i,i);
                return result;
        }
        /**
        * Returns the l(infinity)-norm.
        */
        public double infNorm() {
                double result=0.0,tmpResult;
                for(int j,i=0;i<N;i++) {
                        tmpResult=0.0;
                        for(j=rows[i];j<rows[i+1];j++)
                                tmpResult+=Math.abs(matrix[0][j]);
                        if(tmpResult>result)
                                result=tmpResult;
                }
                return result;
        }
        /**
        * Returns the Frobenius norm.
        */
        public double frobeniusNorm() {
                double result=0.0;
                for(int i=0;i<colPos.length;i++)
                        result+=matrix[0][i]*matrix[0][i];
                return Math.sqrt(result);
        }
        /**
        * Returns the number of rows.
        */
        public int rows() {
                return N;
        }
        /**
        * Returns the number of columns.
        */
        public int columns() {
                return N;
        }

//============
// OPERATIONS
//============

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix add(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        case SPARSE: return add((DoubleSparseMatrix)m);
                        default: 
                                if(N==m.rows() && N==m.columns()) {
                                        final double array[][]=new double[N][N];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=getElement(i,0)+m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=getElement(i,j)+m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleSquareMatrix rawAdd(final DoubleMatrix m) {
                if(N==m.matrix.length && N==m.matrix[0].length) {
                        final double array[][]=new double[N][N];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=getElement(i,0)+m.matrix[i][0];
                                for(j=1;j<array.length;j++)
                                        array[i][j]=getElement(i,j)+m.matrix[i][j];
                        }
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix add(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        case SPARSE: return add((DoubleSparseMatrix)m);
                        default: 
                                if(N==m.rows()) {
                                        final double array[][]=new double[N][N];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=getElement(i,0)+m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=getElement(i,j)+m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a double sparse matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSparseMatrix add(final DoubleSparseMatrix m) {
                if(N==m.rows()) {
                        DoubleSparseMatrix ans=new DoubleSparseMatrix(N);
                        for(int j,i=0;i<N;i++) {
                                ans.setElement(i,0,getElement(i,0)+m.getElement(i,0));
                                for(j=1;j<N;j++)
                                        ans.setElement(i,j,getElement(i,j)+m.getElement(i,j));
                        }
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix subtract(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        case SPARSE: return subtract((DoubleSparseMatrix)m);
                        default: 
                                if(N==m.rows() && N==m.columns()) {
                                        final double array[][]=new double[N][N];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=getElement(i,0)-m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=getElement(i,j)-m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleSquareMatrix rawSubtract(final DoubleMatrix m) {
                if(N==m.matrix.length && N==m.matrix[0].length) {
                        final double array[][]=new double[N][N];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=getElement(i,0)-m.matrix[i][0];
                                for(j=1;j<array.length;j++)
                                        array[i][j]=getElement(i,j)-m.matrix[i][j];
                        }
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix subtract(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        case SPARSE: return subtract((DoubleSparseMatrix)m);
                        default: 
                                if(N==m.rows()) {
                                        final double array[][]=new double[N][N];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=getElement(i,0)-m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=getElement(i,j)-m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a double sparse matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSparseMatrix subtract(final DoubleSparseMatrix m) {
                if(N==m.rows()) {
                        DoubleSparseMatrix ans=new DoubleSparseMatrix(N);
                        for(int j,i=0;i<N;i++) {
                                ans.setElement(i,0,getElement(i,0)-m.getElement(i,0));
                                for(j=1;j<N;j++)
                                        ans.setElement(i,j,getElement(i,j)-m.getElement(i,j));
                        }
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x a double
        * @return a double sparse matrix
        */
        public DoubleMatrix scalarMultiply(final double x) {
                final DoubleSparseMatrix ans=new DoubleSparseMatrix(N);
                ans.matrix[0]=new double[matrix[0].length];
                ans.colPos=new int[colPos.length];
                System.arraycopy(colPos,0,ans.colPos,0,colPos.length);
                System.arraycopy(rows,0,ans.rows,0,rows.length);
                for(int i=0;i<colPos.length;i++)
                        ans.matrix[0][i]=x*matrix[0][i];
                return ans;
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v a double vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public DoubleVector multiply(final DoubleVector v) {
                if(N==v.dimension()) {
                        final double array[]=new double[N];
                        for(int j,i=0;i<array.length;i++) {
                                for(j=rows[i];j<rows[i+1];j++)
                                        array[i]+=matrix[0][j]*v.getComponent(colPos[j]);
                        }
                        return new DoubleVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are incompatible.
        */
        public DoubleMatrix multiply(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        case SPARSE: return multiply((DoubleSparseMatrix)m);
                        default: 
                                if(N==m.rows()) {
                                        int n,k;
                                        final double array[][]=new double[N][m.columns()];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array[0].length;k++) {
                                                        array[j][k]=getElement(j,0)*m.getElement(0,k);
                                                        for(n=1;n<N;n++)
                                                                array[j][k]+=getElement(j,n)*m.getElement(n,k);
                                                }
                                        }
                                        return new DoubleMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private DoubleMatrix rawMultiply(final DoubleMatrix m) {
                if(N==m.matrix.length) {
                        int n,k;
                        final double array[][]=new double[N][m.matrix[0].length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array[0].length;k++) {
                                        array[j][k]=getElement(j,0)*m.matrix[0][k];
                                        for(n=1;n<N;n++)
                                                array[j][k]+=getElement(j,n)*m.matrix[n][k];
                                }
                        }
                        return new DoubleMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are incompatible.
        */
        public DoubleSquareMatrix multiply(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        case SPARSE: return multiply((DoubleSparseMatrix)m);
                        default: 
                                if(N==m.rows()) {
                                        int n,k;
                                        final double array[][]=new double[N][N];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array.length;k++) {
                                                        array[j][k]=getElement(j,0)*m.getElement(0,k);
                                                        for(n=1;n<array.length;n++)
                                                                array[j][k]+=getElement(j,n)*m.getElement(n,k);
                                                }
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private DoubleSquareMatrix rawMultiply(final DoubleSquareMatrix m) {
                if(N==m.matrix.length) {
                        int n,k;
                        final double array[][]=new double[N][N];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array.length;k++) {
                                        array[j][k]=getElement(j,0)*m.matrix[0][k];
                                        for(n=1;n<array.length;n++)
                                                array[j][k]+=getElement(j,n)*m.matrix[n][k];
                                }
                        }
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double sparse matrix
        * @exception MatrixDimensionException If the matrices are incompatible.
        */
        public DoubleSparseMatrix multiply(final DoubleSparseMatrix m) {
                if(N==m.rows()) {
                        int n,k;
                        double tmp;
                        DoubleSparseMatrix ans=new DoubleSparseMatrix(N);
                        for(int j=0;j<N;j++) {
                                for(k=0;k<N;k++) {
                                        tmp=getElement(j,0)*m.getElement(0,k);
                                        for(n=1;n<N;n++)
                                                tmp+=getElement(j,n)*m.getElement(n,k);
                                        ans.setElement(j,k,tmp);
                                }
                        }
                        return ans;
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return a double sparse matrix
        */
        public Matrix transpose() {
                final DoubleSparseMatrix ans=new DoubleSparseMatrix(N);
                for(int j,i=0;i<N;i++) {
                        ans.setElement(0,i,getElement(i,0));
                        for(j=1;j<N;j++)
                                ans.setElement(j,i,getElement(i,j));
                }
                return ans;
        }

// LU DECOMPOSITION

        /**
        * Returns the LU decomposition of this matrix.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] luDecompose() {
                int i,j,k;
                double tmp;
                final double array[][][]=new double[2][N][N];
                array[0][0][0]=1.0;
                for(i=1;i<array[0].length;i++)
                        array[0][i][i]=1.0;
                for(j=0;j<array[0].length;j++) {
                        for(i=0;i<=j;i++) {
                                tmp=getElement(i,j);
                                for(k=0;k<i;k++)
                                        tmp-=array[0][i][k]*array[1][k][j];
                                array[1][i][j]=tmp;
                        }
                        for(i=j+1;i<array[0].length;i++) {
                                tmp=getElement(i,j);
                                for(k=0;k<j;k++)
                                        tmp-=array[0][i][k]*array[1][k][j];
                                array[0][i][j]=tmp/array[1][j][j];
                        }
                }
                final DoubleSquareMatrix lu[]=new DoubleSquareMatrix[2];
                lu[0]=new DoubleSquareMatrix(array[0]);
                lu[1]=new DoubleSquareMatrix(array[1]);
                return lu;
        }

// CHOLESKY DECOMPOSITION

        /**
        * Returns the Cholesky decomposition of this matrix.
        * Matrix must be symmetric and positive definite.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] choleskyDecompose() {
                int i,j,k;
                double tmp;
                final double array[][][]=new double[2][N][N];
                array[0][0][0]=array[1][0][0]=Math.sqrt(getElement(0,0));
                for(i=1;i<array[0].length;i++)
                        array[0][i][0]=array[1][0][i]=getElement(i,0)/array[0][0][0];
                for(j=1;j<array[0].length;j++) {
                        tmp=getElement(j,j);
                        for(i=0;i<j;i++)
                                tmp-=array[0][j][i]*array[0][j][i];
                        array[0][j][j]=array[1][j][j]=Math.sqrt(tmp);
                        for(i=j+1;i<array[0].length;i++) {
                                tmp=getElement(i,j);
                                for(k=0;k<i;k++)
                                        tmp-=array[0][j][k]*array[1][k][i];
                                array[0][i][j]=array[1][j][i]=tmp/array[1][j][j];
                        }
                }
                final DoubleSquareMatrix lu[]=new DoubleSquareMatrix[2];
                lu[0]=new DoubleSquareMatrix(array[0]);
                lu[1]=new DoubleSquareMatrix(array[1]);
                return lu;
        }

// MAP ELEMENTS

        /**
        * Applies a function on all the matrix elements.
        * @param f a user-defined function
        * @return a double sparse matrix
        */
        public DoubleMatrix mapElements(final Mapping f) {
                final DoubleSparseMatrix ans=new DoubleSparseMatrix(N);
                ans.matrix[0]=new double[matrix[0].length];
                ans.colPos=new int[colPos.length];
                System.arraycopy(colPos,0,ans.colPos,0,colPos.length);
                System.arraycopy(rows,0,ans.rows,0,rows.length);
                for(int i=0;i<colPos.length;i++)
                        ans.matrix[0][i]=f.map(matrix[0][i]);
                return ans;
        }
}

