package JSci.maths;

import JSci.GlobalSettings;

/**
* The DoubleTridiagonalMatrix class provides an object for encapsulating tridiagonal matrices containing doubles.
* Uses compressed diagonal storage.
* @version 2.1
* @author Mark Hale
*/
public class DoubleTridiagonalMatrix extends DoubleSquareMatrix {
//
// example: element storage for a 3x3 tridiagonal matrix
//
// matrix[1][0] matrix[2][0]
// matrix[0][1] matrix[1][1] matrix[2][1]
//              matrix[0][2] matrix[1][2]
//
        /**
        * Storage format identifier.
        */
        protected final static int TRIDIAGONAL=3;
        protected final static int storageFormat=TRIDIAGONAL;
        /**
        * Constructs a matrix.
        */
        protected DoubleTridiagonalMatrix() {
                super();
        }
        /**
        * Constructs an empty matrix.
        * @param size the number of rows/columns
        */
        public DoubleTridiagonalMatrix(final int size) {
                this();
                matrix=new double[3][size];
        }
        /**
        * Constructs a matrix from an array.
        * Any non-tridiagonal elements in the array are ignored.
        * @param array an assigned value
        * @exception MatrixDimensionException If the array is not square.
        */
        public DoubleTridiagonalMatrix(final double array[][]) {
                this(array.length);
                if(array.length==array[0].length) {
                        matrix[1][0]=array[0][0];
                        matrix[2][0]=array[0][1];
                        int i=1;
                        for(;i<array.length-1;i++) {
                                matrix[0][i]=array[i][i-1];
                                matrix[1][i]=array[i][i];
                                matrix[2][i]=array[i][i+1];
                        }
                        matrix[0][i]=array[i][i-1];
                        matrix[1][i]=array[i][i];
                } else {
                        matrix=null;
                        throw new MatrixDimensionException("The array is not square.");
                }
        }
        /**
        * Compares two double tridiagonal matrices for equality.
        * @param m a double tridiagonal matrix
        */
        public boolean equals(Object m) {
                if(m!=null && (m instanceof DoubleTridiagonalMatrix) &&
                matrix[1].length==((DoubleTridiagonalMatrix)m).rows()) {
                        final DoubleTridiagonalMatrix dtm=(DoubleTridiagonalMatrix)m;
                        if(Math.abs(matrix[1][0]-dtm.getElement(0,0))>GlobalSettings.ZERO_TOL)
                                return false;
                        if(Math.abs(matrix[2][0]-dtm.getElement(0,1))>GlobalSettings.ZERO_TOL)
                                return false;
                        int i=1;
                        for(;i<matrix[1].length-1;i++) {
                                if(Math.abs(matrix[0][i]-dtm.getElement(i,i-1))>GlobalSettings.ZERO_TOL)
                                        return false;
                                if(Math.abs(matrix[1][i]-dtm.getElement(i,i))>GlobalSettings.ZERO_TOL)
                                        return false;
                                if(Math.abs(matrix[2][i]-dtm.getElement(i,i+1))>GlobalSettings.ZERO_TOL)
                                        return false;
                        }
                        if(Math.abs(matrix[0][i]-dtm.getElement(i,i-1))>GlobalSettings.ZERO_TOL)
                                return false;
                        if(Math.abs(matrix[1][i]-dtm.getElement(i,i))>GlobalSettings.ZERO_TOL)
                                return false;
                        return true;
                } else {
                        return false;
                }
        }
        /**
        * Returns a string representing this matrix.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer(matrix.length*matrix[1].length);
                for(int j,i=0;i<rows();i++) {
                        for(j=0;j<columns();j++) {
                                buf.append(getElement(i,j));
                                buf.append(' ');
                        }
                        buf.append('\n');
                }
                return buf.toString();
        }
        /**
        * Converts this matrix to an integer matrix.
        * @return an integer tridiagonal matrix
        */
        public IntegerMatrix toIntegerMatrix() {
                final IntegerTridiagonalMatrix m=new IntegerTridiagonalMatrix(matrix[1].length);
                m.matrix[1][0]=Math.round((float)matrix[1][0]);
                m.matrix[2][0]=Math.round((float)matrix[2][0]);
                int i=1;
                for(;i<matrix[1].length-1;i++) {
                        m.matrix[0][i]=Math.round((float)matrix[0][i]);
                        m.matrix[1][i]=Math.round((float)matrix[1][i]);
                        m.matrix[2][i]=Math.round((float)matrix[2][i]);
                }
                m.matrix[0][i]=Math.round((float)matrix[0][i]);
                m.matrix[1][i]=Math.round((float)matrix[1][i]);
                return m;
        }
        /**
        * Converts this matrix to a complex matrix.
        * @return a complex tridiagonal matrix
        */
        public ComplexMatrix toComplexMatrix() {
                final ComplexTridiagonalMatrix m=new ComplexTridiagonalMatrix(matrix[1].length);
                m.matrix[1][0]=new Complex(matrix[1][0],0.0);
                m.matrix[2][0]=new Complex(matrix[2][0],0.0);
                int i=1;
                for(;i<matrix[1].length-1;i++) {
                        m.matrix[0][i]=new Complex(matrix[0][i],0.0);
                        m.matrix[1][i]=new Complex(matrix[1][i],0.0);
                        m.matrix[2][i]=new Complex(matrix[2][i],0.0);
                }
                m.matrix[0][i]=new Complex(matrix[0][i],0.0);
                m.matrix[1][i]=new Complex(matrix[1][i],0.0);
                return m;
        }
        /**
        * Returns an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public double getElement(final int i, final int j) {
                if(i>=0 && i<matrix[1].length && j>=0 && j<matrix[1].length) {
                        if(j==i-1)
                                return matrix[0][i];
                        else if(j==i)
                                return matrix[1][i];
                        else if(j==i+1)
                                return matrix[2][i];
                        else
                                return 0.0;
                } else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Sets the value of an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @param x a number
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public void setElement(final int i, final int j, final double x) {
                if(i>=0 && i<matrix[1].length && j>=0 && j<matrix[1].length) {
                        if(j==i-1)
                                matrix[0][i]=x;
                        else if(j==i)
                                matrix[1][i]=x;
                        else if(j==i+1)
                                matrix[2][i]=x;
                        else
                                throw new MatrixDimensionException("Invalid element.");
                } else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Returns true if this matrix is symmetric.
        */
        public boolean isSymmetric() {
                if(matrix[0][1]!=matrix[2][0])
                        return false;
                for(int i=1;i<matrix[1].length-1;i++) {
                        if(matrix[0][i+1]!=matrix[2][i])
                                return false;
                }
                return true;
        }
        /**
        * Returns the trace.
        */
        public double trace() {
                double tr=matrix[1][0];
                for(int i=1;i<matrix[1].length;i++)
                        tr+=matrix[1][i];
                return tr;
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public double infNorm() {
                double result=Math.abs(matrix[1][0])+Math.abs(matrix[2][0]);
                double tmpResult;
                int i=1;
                for(;i<matrix[1].length-1;i++) {
                        tmpResult=Math.abs(matrix[0][i])+Math.abs(matrix[1][i])+Math.abs(matrix[2][i]);
                        if(tmpResult>result)
                                result=tmpResult;
                }
                tmpResult=Math.abs(matrix[0][i])+Math.abs(matrix[1][i]);
                if(tmpResult>result)
                        result=tmpResult;
                return result;
        }
        /**
        * Returns the Frobenius norm.
        * @author Taber Smith
        */
        public double frobeniusNorm() {
                double result=matrix[1][0]*matrix[1][0]+matrix[2][0]*matrix[2][0];
                int i=1;
                for(;i<matrix[1].length-1;i++)
                        result+=matrix[0][i]*matrix[0][i]+matrix[1][i]*matrix[1][i]+matrix[2][i]*matrix[2][i];
                result+=matrix[0][i]*matrix[0][i]+matrix[1][i]*matrix[1][i];
                return Math.sqrt(result);
        }
        /**
        * Returns the number of rows.
        */
        public int rows() {
                return matrix[1].length;
        }
        /**
        * Returns the number of columns.
        */
        public int columns() {
                return matrix[1].length;
        }

//============
// OPERATIONS
//============

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix add(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        case TRIDIAGONAL: return rawAdd((DoubleTridiagonalMatrix)m);
                        default: 
                                if(matrix[1].length==m.rows() && matrix[1].length==m.columns()) {
                                        final double array[][]=new double[matrix[1].length][matrix[1].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=getElement(i,0)+m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=getElement(i,j)+m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleSquareMatrix rawAdd(final DoubleMatrix m) {
                if(matrix[1].length==m.matrix.length && matrix[1].length==m.matrix[0].length) {
                        final double array[][]=new double[matrix[1].length][matrix[1].length];
                        for(int i=0;i<array.length;i++)
                                System.arraycopy(m.matrix[i],0,array[i],0,array.length);
                        array[0][0]+=matrix[1][0];
                        array[0][1]+=matrix[2][0];
                        int n=array.length-1;
                        for(int i=1;i<n;i++) {
                                array[i][i-1]+=matrix[0][i];
                                array[i][i]+=matrix[1][i];
                                array[i][i+1]+=matrix[2][i];
                        }
                        array[n][n-1]+=matrix[0][n];
                        array[n][n]+=matrix[1][n];
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix add(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawAdd(m);
                        case TRIDIAGONAL: return rawAdd((DoubleTridiagonalMatrix)m);
                        default: 
                                if(matrix[1].length==m.rows()) {
                                        final double array[][]=new double[matrix[1].length][matrix[1].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=getElement(i,0)+m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=getElement(i,j)+m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m a double tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleTridiagonalMatrix add(final DoubleTridiagonalMatrix m) {
                switch(m.storageFormat) {
                        case TRIDIAGONAL: return rawAdd(m);
                        default: 
                                int mRow=matrix[1].length;
                                if(mRow==m.rows()) {
                                        final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(mRow);
                                        ans.matrix[1][0]=matrix[1][0]+m.getElement(0,0);
                                        ans.matrix[2][0]=matrix[2][0]+m.getElement(0,1);
                                        mRow--;
                                        for(int i=1;i<mRow;i++) {
                                                ans.matrix[0][i]=matrix[0][i]+m.getElement(i,i-1);
                                                ans.matrix[1][i]=matrix[1][i]+m.getElement(i,i);
                                                ans.matrix[2][i]=matrix[2][i]+m.getElement(i,i+1);
                                        }
                                        ans.matrix[0][mRow]=matrix[0][mRow]+m.getElement(mRow,mRow-1);
                                        ans.matrix[1][mRow]=matrix[1][mRow]+m.getElement(mRow,mRow);
                                        return ans;
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleTridiagonalMatrix rawAdd(final DoubleTridiagonalMatrix m) {
                int mRow=matrix[1].length;
                if(mRow==m.matrix[1].length) {
                        final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(mRow);
                        ans.matrix[1][0]=matrix[1][0]+m.matrix[1][0];
                        ans.matrix[2][0]=matrix[2][0]+m.matrix[2][0];
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.matrix[0][i]=matrix[0][i]+m.matrix[0][i];
                                ans.matrix[1][i]=matrix[1][i]+m.matrix[1][i];
                                ans.matrix[2][i]=matrix[2][i]+m.matrix[2][i];
                        }
                        ans.matrix[0][mRow]=matrix[0][mRow]+m.matrix[0][mRow];
                        ans.matrix[1][mRow]=matrix[1][mRow]+m.matrix[1][mRow];
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix subtract(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        case TRIDIAGONAL: return rawSubtract((DoubleTridiagonalMatrix)m);
                        default: 
                                if(matrix[1].length==m.rows() && matrix[1].length==m.columns()) {
                                        final double array[][]=new double[matrix[1].length][matrix[1].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=getElement(i,0)-m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=getElement(i,j)-m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleSquareMatrix rawSubtract(final DoubleMatrix m) {
                if(matrix[1].length==m.matrix.length && matrix[1].length==m.matrix[0].length) {
                        final double array[][]=new double[matrix[1].length][matrix[1].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=getElement(i,0)-m.matrix[i][0];
                                for(j=1;j<array.length;j++)
                                        array[i][j]=getElement(i,j)-m.matrix[i][j];
                        }
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix subtract(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawSubtract(m);
                        case TRIDIAGONAL: return rawSubtract((DoubleTridiagonalMatrix)m);
                        default: 
                                if(matrix[1].length==m.rows()) {
                                        final double array[][]=new double[matrix[1].length][matrix[1].length];
                                        for(int j,i=0;i<array.length;i++) {
                                                array[i][0]=getElement(i,0)-m.getElement(i,0);
                                                for(j=1;j<array.length;j++)
                                                        array[i][j]=getElement(i,j)-m.getElement(i,j);
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        /**
        * Returns the subtraction of this matrix and another.
        * @param m a double tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleTridiagonalMatrix subtract(final DoubleTridiagonalMatrix m) {
                switch(m.storageFormat) {
                        case TRIDIAGONAL: return rawSubtract(m);
                        default: 
                                int mRow=matrix[1].length;
                                if(mRow==m.rows()) {
                                        final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(mRow);
                                        ans.matrix[1][0]=matrix[1][0]-m.getElement(0,0);
                                        ans.matrix[2][0]=matrix[2][0]-m.getElement(0,1);
                                        mRow--;
                                        for(int i=1;i<mRow;i++) {
                                                ans.matrix[0][i]=matrix[0][i]-m.getElement(i,i-1);
                                                ans.matrix[1][i]=matrix[1][i]-m.getElement(i,i);
                                                ans.matrix[2][i]=matrix[2][i]-m.getElement(i,i+1);
                                        }
                                        ans.matrix[0][mRow]=matrix[0][mRow]-m.getElement(mRow,mRow-1);
                                        ans.matrix[1][mRow]=matrix[1][mRow]-m.getElement(mRow,mRow);
                                        return ans;
                                } else
                                        throw new MatrixDimensionException("Matrices are different sizes.");
                }
        }
        private DoubleTridiagonalMatrix rawSubtract(final DoubleTridiagonalMatrix m) {
                int mRow=matrix[1].length;
                if(mRow==m.matrix[1].length) {
                        final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(mRow);
                        ans.matrix[1][0]=matrix[1][0]-m.matrix[1][0];
                        ans.matrix[2][0]=matrix[2][0]-m.matrix[2][0];
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.matrix[0][i]=matrix[0][i]-m.matrix[0][i];
                                ans.matrix[1][i]=matrix[1][i]-m.matrix[1][i];
                                ans.matrix[2][i]=matrix[2][i]-m.matrix[2][i];
                        }
                        ans.matrix[0][mRow]=matrix[0][mRow]-m.matrix[0][mRow];
                        ans.matrix[1][mRow]=matrix[1][mRow]-m.matrix[1][mRow];
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x a double
        * @return a double tridiagonal matrix
        */
        public DoubleMatrix scalarMultiply(final double x) {
                int mRow=matrix[1].length;
                final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(mRow);
                ans.matrix[1][0]=x*matrix[1][0];
                ans.matrix[2][0]=x*matrix[2][0];
                mRow--;
                for(int i=1;i<mRow;i++) {
                        ans.matrix[0][i]=x*matrix[0][i];
                        ans.matrix[1][i]=x*matrix[1][i];
                        ans.matrix[2][i]=x*matrix[2][i];
                }
                ans.matrix[0][mRow]=x*matrix[0][mRow];
                ans.matrix[1][mRow]=x*matrix[1][mRow];
                return ans;
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v a double vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public DoubleVector multiply(final DoubleVector v) {
                int mRow=matrix[1].length;
                if(mRow==v.dimension()) {
                        final double array[]=new double[mRow];
                        array[0]=matrix[1][0]*v.getComponent(0)+matrix[2][0]*v.getComponent(1);
                        mRow--;
                        for(int i=1;i<mRow;i++)
                                array[i]=matrix[0][i]*v.getComponent(i-1)+matrix[1][i]*v.getComponent(i)+matrix[2][i]*v.getComponent(i+1);
                        array[mRow]=matrix[0][mRow]*v.getComponent(mRow-1)+matrix[1][mRow]*v.getComponent(mRow);
                        return new DoubleVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleMatrix multiply(final DoubleMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        case TRIDIAGONAL: return rawMultiply((DoubleTridiagonalMatrix)m);
                        default: 
                                if(matrix[1].length==m.rows()) {
                                        int n,k;
                                        final double array[][]=new double[matrix[1].length][m.columns()];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array[0].length;k++) {
                                                        array[j][k]=getElement(j,0)*m.getElement(0,k);
                                                        for(n=1;n<matrix[1].length;n++)
                                                                array[j][k]+=getElement(j,n)*m.getElement(n,k);
                                                }
                                        }
                                        return new DoubleMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private DoubleMatrix rawMultiply(final DoubleMatrix m) {
                if(matrix[1].length==m.matrix.length) {
                        int n,k;
                        final double array[][]=new double[matrix[1].length][m.matrix[0].length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array[0].length;k++) {
                                        array[j][k]=getElement(j,0)*m.matrix[0][k];
                                        for(n=1;n<matrix[1].length;n++)
                                                array[j][k]+=getElement(j,n)*m.matrix[n][k];
                                }
                        }
                        return new DoubleMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix multiply(final DoubleSquareMatrix m) {
                switch(m.storageFormat) {
                        case ARRAY_2D: return rawMultiply(m);
                        case TRIDIAGONAL: return rawMultiply((DoubleTridiagonalMatrix)m);
                        default: 
                                if(matrix[1].length==m.rows()) {
                                        int n,k;
                                        final double array[][]=new double[matrix[1].length][matrix[1].length];
                                        for(int j=0;j<array.length;j++) {
                                                for(k=0;k<array.length;k++) {
                                                        array[j][k]=getElement(j,0)*m.getElement(0,k);
                                                        for(n=1;n<array.length;n++)
                                                                array[j][k]+=getElement(j,n)*m.getElement(n,k);
                                                }
                                        }
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private DoubleSquareMatrix rawMultiply(final DoubleSquareMatrix m) {
                if(matrix[1].length==m.matrix.length) {
                        int n,k;
                        final double array[][]=new double[matrix[1].length][matrix[1].length];
                        for(int j=0;j<array.length;j++) {
                                for(k=0;k<array.length;k++) {
                                        array[j][k]=getElement(j,0)*m.matrix[0][k];
                                        for(n=1;n<array.length;n++)
                                                array[j][k]+=getElement(j,n)*m.matrix[n][k];
                                }
                        }
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m a double tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public DoubleSquareMatrix multiply(final DoubleTridiagonalMatrix m) {
                switch(m.storageFormat) {
                        case TRIDIAGONAL: return rawMultiply(m);
                        default: 
                                int mRow=matrix[1].length;
                                if(mRow==m.rows()) {
                                        final double array[][]=new double[mRow][mRow];
                                        array[0][0]=matrix[1][0]*m.getElement(0,0)+matrix[2][0]*m.getElement(1,0);
                                        array[0][1]=matrix[1][0]*m.getElement(0,1)+matrix[2][0]*m.getElement(1,1);
                                        array[0][2]=matrix[2][0]*m.getElement(1,2);
                                        if(mRow>3) {
                                                array[1][0]=matrix[0][1]*m.getElement(0,0)+matrix[1][1]*m.getElement(1,0);
                                                array[1][1]=matrix[0][1]*m.getElement(0,1)+matrix[1][1]*m.getElement(1,1)+matrix[2][1]*m.getElement(2,1);
                                                array[1][2]=matrix[1][1]*m.getElement(1,2)+matrix[2][1]*m.getElement(2,2);
                                                array[1][3]=matrix[2][1]*m.getElement(2,3);
                                        }
                                        if(mRow==3) {
                                                array[1][0]=matrix[0][1]*m.getElement(0,0)+matrix[1][1]*m.getElement(1,0);
                                                array[1][1]=matrix[0][1]*m.getElement(0,1)+matrix[1][1]*m.getElement(1,1)+matrix[2][1]*m.getElement(2,1);
                                                array[1][2]=matrix[1][1]*m.getElement(1,2)+matrix[2][1]*m.getElement(2,2);
                                        } else if(mRow>4) {
                                                for(int i=2;i<mRow-2;i++) {
                                                        array[i][i-2]=matrix[0][i]*m.getElement(i-1,i-2);
                                                        array[i][i-1]=matrix[0][i]*m.getElement(i-1,i-1)+matrix[1][i]*m.getElement(i,i-1);
                                                        array[i][i]=matrix[0][i]*m.getElement(i-1,i)+matrix[1][i]*m.getElement(i,i)+matrix[2][i]*m.getElement(i+1,i);
                                                        array[i][i+1]=matrix[1][i]*m.getElement(i,i+1)+matrix[2][i]*m.getElement(i+1,i+1);
                                                        array[i][i+2]=matrix[2][i]*m.getElement(i+1,i+2);
                                                }
                                        }
                                        if(mRow>3) {
                                                array[mRow-2][mRow-4]=matrix[0][mRow-2]*m.getElement(mRow-3,mRow-4);
                                                array[mRow-2][mRow-3]=matrix[0][mRow-2]*m.getElement(mRow-3,mRow-3)+matrix[1][mRow-2]*m.getElement(mRow-2,mRow-3);
                                                array[mRow-2][mRow-2]=matrix[0][mRow-2]*m.getElement(mRow-3,mRow-2)+matrix[1][mRow-2]*m.getElement(mRow-2,mRow-2)+matrix[2][mRow-2]*m.getElement(mRow-1,mRow-2);
                                                array[mRow-2][mRow-1]=matrix[1][mRow-2]*m.getElement(mRow-2,mRow-1)+matrix[2][mRow-2]*m.getElement(mRow-1,mRow-1);
                                        }
                                        mRow--;
                                        array[mRow][mRow-2]=matrix[0][mRow]*m.getElement(mRow-1,mRow-2);
                                        array[mRow][mRow-1]=matrix[0][mRow]*m.getElement(mRow-1,mRow-1)+matrix[1][mRow]*m.getElement(mRow,mRow-1);
                                        array[mRow][mRow]=matrix[0][mRow]*m.getElement(mRow-1,mRow)+matrix[1][mRow]*m.getElement(mRow,mRow);
                                        return new DoubleSquareMatrix(array);
                                } else
                                        throw new MatrixDimensionException("Incompatible matrices.");
                }
        }
        private DoubleSquareMatrix rawMultiply(final DoubleTridiagonalMatrix m) {
                int mRow=matrix[1].length;
                if(mRow==m.matrix[1].length) {
                        final double array[][]=new double[mRow][mRow];
                        array[0][0]=matrix[1][0]*m.matrix[1][0]+matrix[2][0]*m.matrix[0][1];
                        array[0][1]=matrix[1][0]*m.matrix[2][0]+matrix[2][0]*m.matrix[1][1];
                        array[0][2]=matrix[2][0]*m.matrix[2][1];
                        if(mRow>3) {
                                array[1][0]=matrix[0][1]*m.matrix[1][0]+matrix[1][1]*m.matrix[0][1];
                                array[1][1]=matrix[0][1]*m.matrix[2][0]+matrix[1][1]*m.matrix[1][1]+matrix[2][1]*m.matrix[0][2];
                                array[1][2]=matrix[1][1]*m.matrix[2][1]+matrix[2][1]*m.matrix[1][2];
                                array[1][3]=matrix[2][1]*m.matrix[2][2];
                        }
                        if(mRow==3) {
                                array[1][0]=matrix[0][1]*m.matrix[1][0]+matrix[1][1]*m.matrix[0][1];
                                array[1][1]=matrix[0][1]*m.matrix[2][0]+matrix[1][1]*m.matrix[1][1]+matrix[2][1]*m.matrix[0][2];
                                array[1][2]=matrix[1][1]*m.matrix[2][1]+matrix[2][1]*m.matrix[1][2];
                        } else if(mRow>4) {
                                for(int i=2;i<mRow-2;i++) {
                                        array[i][i-2]=matrix[0][i]*m.matrix[0][i-1];
                                        array[i][i-1]=matrix[0][i]*m.matrix[1][i-1]+matrix[1][i]*m.matrix[0][i];
                                        array[i][i]=matrix[0][i]*m.matrix[2][i-1]+matrix[1][i]*m.matrix[1][i]+matrix[2][i]*m.matrix[0][i+1];
                                        array[i][i+1]=matrix[1][i]*m.matrix[2][i]+matrix[2][i]*m.matrix[1][i+1];
                                        array[i][i+2]=matrix[2][i]*m.matrix[2][i+1];
                                }
                        }
                        if(mRow>3) {
                                array[mRow-2][mRow-4]=matrix[0][mRow-2]*m.matrix[0][mRow-3];
                                array[mRow-2][mRow-3]=matrix[0][mRow-2]*m.matrix[1][mRow-3]+matrix[1][mRow-2]*m.matrix[0][mRow-2];
                                array[mRow-2][mRow-2]=matrix[0][mRow-2]*m.matrix[2][mRow-3]+matrix[1][mRow-2]*m.matrix[1][mRow-2]+matrix[2][mRow-2]*m.matrix[0][mRow-1];
                                array[mRow-2][mRow-1]=matrix[1][mRow-2]*m.matrix[2][mRow-2]+matrix[2][mRow-2]*m.matrix[1][mRow-1];
                        }
                        mRow--;
                        array[mRow][mRow-2]=matrix[0][mRow]*m.matrix[0][mRow-1];
                        array[mRow][mRow-1]=matrix[0][mRow]*m.matrix[1][mRow-1]+matrix[1][mRow]*m.matrix[0][mRow];
                        array[mRow][mRow]=matrix[0][mRow]*m.matrix[2][mRow-1]+matrix[1][mRow]*m.matrix[1][mRow];
                        return new DoubleSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Incompatible matrices.");
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return a double tridiagonal matrix
        */
        public Matrix transpose() {
                final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(matrix[1].length);
                System.arraycopy(matrix[0],1,ans.matrix[2],0,matrix[0].length-1);
                System.arraycopy(matrix[1],0,ans.matrix[1],0,matrix[1].length);
                System.arraycopy(matrix[2],0,ans.matrix[0],1,matrix[2].length-1);
                return ans;
        }

// LU DECOMPOSITION

        /**
        * Returns the LU decomposition of this matrix.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] luDecompose() {
                int i,j,k;
                double tmp;
                final double array[][][]=new double[2][matrix[1].length][matrix[1].length];
                array[0][0][0]=1.0;
                for(i=1;i<matrix[1].length;i++)
                        array[0][i][i]=1.0;
                for(j=0;j<matrix[1].length;j++) {
                        for(i=0;i<=j;i++) {
                                tmp=getElement(i,j);
                                for(k=0;k<i;k++)
                                        tmp-=array[0][i][k]*array[1][k][j];
                                array[1][i][j]=tmp;
                        }
                        for(i=j+1;i<matrix[1].length;i++) {
                                tmp=getElement(i,j);
                                for(k=0;k<j;k++)
                                        tmp-=array[0][i][k]*array[1][k][j];
                                array[0][i][j]=tmp/array[1][j][j];
                        }
                }
                final DoubleSquareMatrix lu[]=new DoubleSquareMatrix[2];
                lu[0]=new DoubleSquareMatrix(array[0]);
                lu[1]=new DoubleSquareMatrix(array[1]);
                return lu;
        }

// CHOLESKY DECOMPOSITION

        /**
        * Returns the Cholesky decomposition of this matrix.
        * Matrix must be symmetric and positive definite.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] choleskyDecompose() {
                int i,j,k;
                double tmp;
                final double array[][][]=new double[2][matrix[1].length][matrix[1].length];
                array[0][0][0]=array[1][0][0]=Math.sqrt(getElement(0,0));
                for(i=1;i<matrix[1].length;i++)
                        array[0][i][0]=array[1][0][i]=getElement(i,0)/array[0][0][0];
                for(j=1;j<matrix[1].length;j++) {
                        tmp=getElement(j,j);
                        for(i=0;i<j;i++)
                                tmp-=array[0][j][i]*array[0][j][i];
                        array[0][j][j]=array[1][j][j]=Math.sqrt(tmp);
                        for(i=j+1;i<matrix[1].length;i++) {
                                tmp=getElement(i,j);
                                for(k=0;k<i;k++)
                                        tmp-=array[0][j][k]*array[1][k][i];
                                array[0][i][j]=array[1][j][i]=tmp/array[1][j][j];
                        }
                }
                final DoubleSquareMatrix lu[]=new DoubleSquareMatrix[2];
                lu[0]=new DoubleSquareMatrix(array[0]);
                lu[1]=new DoubleSquareMatrix(array[1]);
                return lu;
        }

// MAP ELEMENTS

        /**
        * Applies a function on all the matrix elements.
        * @param f a user-defined function
        * @return a double tridiagonal matrix
        */
        public DoubleMatrix mapElements(final Mapping f) {
                int mRow=matrix[1].length;
                final DoubleTridiagonalMatrix ans=new DoubleTridiagonalMatrix(mRow);
                ans.matrix[1][0]=f.map(matrix[1][0]);
                ans.matrix[2][0]=f.map(matrix[2][0]);
                mRow--;
                for(int i=1;i<mRow;i++) {
                        ans.matrix[0][i]=f.map(matrix[0][i]);
                        ans.matrix[1][i]=f.map(matrix[1][i]);
                        ans.matrix[2][i]=f.map(matrix[2][i]);
                }
                ans.matrix[0][mRow]=f.map(matrix[0][mRow]);
                ans.matrix[1][mRow]=f.map(matrix[1][mRow]);
                return ans;
        }
}

