package JSci.maths;

/**
* The IntegerDiagonalMatrix class provides an object for encapsulating diagonal matrices containing integers.
* Uses compressed diagonal storage.
* @version 2.1
* @author Mark Hale
*/
public final class IntegerDiagonalMatrix extends IntegerTridiagonalMatrix {
        /**
        * Storage format identifier.
        */
        protected final static int DIAGONAL=4;
        protected final static int storageFormat=DIAGONAL;
        /**
        * Constructs a matrix.
        */
        protected IntegerDiagonalMatrix() {
                super();
        }
        /**
        * Constructs an empty matrix.
        * @param size the number of rows/columns
        */
        public IntegerDiagonalMatrix(final int size) {
                this();
                matrix=new int[1][size];
        }
        /**
        * Constructs a matrix from an array.
        * @param array an assigned value
        * @exception MatrixDimensionException If the array is not square.
        */
        public IntegerDiagonalMatrix(final int array[][]) {
                this(array.length);
                if(array.length==array[0].length) {
                        for(int i=0;i<matrix[0].length;i++)
                                matrix[0][i]=array[i][i];
                } else {
                        matrix=null;
                        throw new MatrixDimensionException("Array must be square.");
                }
        }
        /**
        * Constructs a matrix by wrapping an array containing the diagonal elements.
        * @param array an assigned value
        */
        public IntegerDiagonalMatrix(final int array[]) {
                this();
                matrix=new int[1][];
                matrix[0]=array;
        }
        /**
        * Creates an identity matrix.
        * @param size the number of rows/columns
        */
        public static IntegerDiagonalMatrix identity(final int size) {
                final int array[]=new int[size];
                for(int i=0;i<size;i++)
                        array[i]=1;
                return new IntegerDiagonalMatrix(array);
        }
        /**
        * Compares two integer diagonal matrices for equality.
        * @param m a integer diagonal matrix
        */
        public boolean equals(Object m) {
                if(m!=null && (m instanceof IntegerDiagonalMatrix) &&
                matrix[0].length==((IntegerDiagonalMatrix)m).rows()) {
                        final IntegerDiagonalMatrix idm=(IntegerDiagonalMatrix)m;
                        for(int i=0;i<matrix[0].length;i++) {
                                if(matrix[0][i]!=idm.getElement(i,i))
                                        return false;
                        }
                        return true;
                } else
                        return false;
        }
        /**
        * Converts this matrix to a double matrix.
        * @return a double diagonal matrix
        */
        public DoubleMatrix toDoubleMatrix() {
                final double array[]=new double[matrix[0].length];
                for(int i=0;i<matrix[0].length;i++)
                        array[i]=matrix[0][i];
                return new DoubleDiagonalMatrix(array);
        }
        /**
        * Converts this matrix to a complex matrix.
        * @return a complex diagonal matrix
        */
        public ComplexMatrix toComplexMatrix() {
                final Complex array[]=new Complex[matrix[0].length];
                for(int i=0;i<matrix[0].length;i++)
                        array[i]=new Complex(matrix[0][i],0.0);
                return new ComplexDiagonalMatrix(array);
        }
        /**
        * Returns an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public int getElement(final int i, final int j) {
                if(i>=0 && i<matrix[0].length && j>=0 && j<matrix[0].length) {
                        if(i==j)
                                return matrix[0][i];
                        else
                                return 0;
                } else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Sets the value of an element of the matrix.
        * @param i row index of the element
        * @param j column index of the element
        * @param x an integer
        * @exception MatrixDimensionException If attempting to access an invalid element.
        */
        public void setElement(final int i, final int j, final int x) {
                if(i>=0 && i<matrix[0].length && j>=0 && j<matrix[0].length && i==j)
                        matrix[0][i]=x;
                else
                        throw new MatrixDimensionException("Invalid element.");
        }
        /**
        * Returns true if this matrix is symmetric.
        */
        public boolean isSymmetric() {
                return true;
        }
        /**
        * Returns true if this matrix is unitary.
        */
        public boolean isUnitary() {
                return this.multiply(this).equals(identity(matrix[0].length));
        }
        /**
        * Returns the determinant.
        */
        public int det() {
                int det=matrix[0][0];
                for(int i=1;i<matrix[0].length;i++)
                        det*=matrix[0][i];
                return det;
        }
        /**
        * Returns the trace.
        */
        public int trace() {
                int tr=matrix[0][0];
                for(int i=1;i<matrix[0].length;i++)
                        tr+=matrix[0][i];
                return tr;
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public int infNorm() {
                int result=Math.abs(matrix[0][0]);
                int tmpResult;
                for(int i=1;i<matrix[0].length;i++) {
                        tmpResult=Math.abs(matrix[0][i]);
                        if(tmpResult>result)
                                result=tmpResult;
                }
                return result;
        }
        /**
        * Returns the Frobenius norm.
        * @author Taber Smith
        */
        public double frobeniusNorm() {
                int result=matrix[0][0]*matrix[0][0];
                for(int i=1;i<matrix[0].length;i++)
                        result+=matrix[0][i]*matrix[0][i];
                return Math.sqrt(result);
        }
        /**
        * Returns the number of rows.
        */
        public int rows() {
                return matrix[0].length;
        }
        /**
        * Returns the number of columns.
        */
        public int columns() {
                return matrix[0].length;
        }

//============
// OPERATIONS
//============

// ADDITION

        /**
        * Returns the addition of this matrix and another.
        * @param m an integer matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerMatrix add(final IntegerMatrix m) {
                if(m instanceof IntegerDiagonalMatrix)
                        return add((IntegerDiagonalMatrix)m);
                else if(m instanceof IntegerTridiagonalMatrix)
                        return rawAdd((IntegerTridiagonalMatrix)m);
                else if(m instanceof IntegerSquareMatrix)
                        return rawAdd((IntegerSquareMatrix)m);
                else
                        return rawAdd(m);
        }
        private IntegerMatrix rawAdd(final IntegerMatrix m) {
                if(matrix[0].length==m.rows() && matrix[0].length==m.columns()) {
                        final int array[][]=new int[matrix[0].length][matrix[0].length];
                        for(int i=0;i<array.length;i++)
                                System.arraycopy(m.matrix[i],0,array[i],0,array.length);
                        for(int i=0;i<array.length;i++)
                                array[i][i]+=matrix[0][i];
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m an integer square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerSquareMatrix add(final IntegerSquareMatrix m) {
                if(m instanceof IntegerDiagonalMatrix)
                        return add((IntegerDiagonalMatrix)m);
                else if(m instanceof IntegerTridiagonalMatrix)
                        return rawAdd((IntegerTridiagonalMatrix)m);
                else
                        return rawAdd(m);
        }
        private IntegerSquareMatrix rawAdd(final IntegerSquareMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final int array[][]=new int[matrix[0].length][matrix[0].length];
                        for(int i=0;i<array.length;i++)
                                System.arraycopy(m.matrix[i],0,array[i],0,array.length);
                        for(int i=0;i<array.length;i++)
                                array[i][i]+=matrix[0][i];
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m an integer tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerTridiagonalMatrix add(final IntegerTridiagonalMatrix m) {
                if(m instanceof IntegerDiagonalMatrix)
                        return add((IntegerDiagonalMatrix)m);
                else
                        return rawAdd(m);
        }
        private IntegerTridiagonalMatrix rawAdd(final IntegerTridiagonalMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final IntegerTridiagonalMatrix ans=new IntegerTridiagonalMatrix(matrix[0].length);
                        System.arraycopy(m.matrix[0],0,ans.matrix[0],0,matrix[0].length);
                        System.arraycopy(m.matrix[2],0,ans.matrix[2],0,matrix[2].length);
                        ans.matrix[1][0]=matrix[0][0]+m.matrix[1][0];
                        for(int i=1;i<matrix[0].length;i++)
                                ans.matrix[1][i]=matrix[0][i]+m.matrix[1][i];
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the addition of this matrix and another.
        * @param m an integer diagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerDiagonalMatrix add(final IntegerDiagonalMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final int array[]=new int[matrix[0].length];
                        array[0]=matrix[0][0]+m.getElement(0,0);
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i]+m.getElement(i,i);
                        return new IntegerDiagonalMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this matrix and another.
        * @param m an integer matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerMatrix subtract(final IntegerMatrix m) {
                if(m instanceof IntegerDiagonalMatrix)
                        return subtract((IntegerDiagonalMatrix)m);
                else if(m instanceof IntegerTridiagonalMatrix)
                        return rawSubtract((IntegerTridiagonalMatrix)m);
                else if(m instanceof IntegerSquareMatrix)
                        return rawSubtract((IntegerSquareMatrix)m);
                else
                        return rawSubtract(m);
        }
        private IntegerMatrix rawSubtract(final IntegerMatrix m) {
                if(matrix[0].length==m.rows() && matrix[0].length==m.columns()) {
                        final int array[][]=new int[matrix[0].length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=-m.matrix[i][0];
                                for(j=1;j<array.length;j++)
                                        array[i][j]=-m.matrix[i][j];
                        }
                        for(int i=0;i<array.length;i++)
                                array[i][i]+=matrix[0][i];
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix and another.
        * @param m an integer square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerSquareMatrix subtract(final IntegerSquareMatrix m) {
                if(m instanceof IntegerDiagonalMatrix)
                        return subtract((IntegerDiagonalMatrix)m);
                else if(m instanceof IntegerTridiagonalMatrix)
                        return rawSubtract((IntegerTridiagonalMatrix)m);
                else
                        return rawSubtract(m);
        }
        private IntegerSquareMatrix rawSubtract(final IntegerSquareMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final int array[][]=new int[matrix[0].length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=-m.matrix[i][0];
                                for(j=1;j<array.length;j++)
                                        array[i][j]=-m.matrix[i][j];
                        }
                        for(int i=0;i<array.length;i++)
                                array[i][i]+=matrix[0][i];
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix and another.
        * @param m an integer tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerTridiagonalMatrix subtract(final IntegerTridiagonalMatrix m) {
                if(m instanceof IntegerDiagonalMatrix)
                        return subtract((IntegerDiagonalMatrix)m);
                else
                        return rawSubtract(m);
        }
        private IntegerTridiagonalMatrix rawSubtract(final IntegerTridiagonalMatrix m) {
                if(matrix[0].length==m.rows()) {
                        int mRow=matrix[0].length;
                        final IntegerTridiagonalMatrix ans=new IntegerTridiagonalMatrix(mRow);
                        ans.setElement(0,0,matrix[0][0]-m.getElement(0,0));
                        ans.setElement(1,0,-m.getElement(1,0));
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.setElement(i-1,i,-m.getElement(i-1,i));
                                ans.setElement(i,i,matrix[0][i]-m.getElement(i,i));
                                ans.setElement(i+1,i,-m.getElement(i+1,i));
                        }
                        ans.setElement(mRow-1,mRow,-m.getElement(mRow-1,mRow));
                        ans.setElement(mRow,mRow,matrix[0][mRow]-m.getElement(mRow,mRow));
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the subtraction of this matrix by another.
        * @param m an integer diagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerDiagonalMatrix subtract(final IntegerDiagonalMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final int array[]=new int[matrix[0].length];
                        array[0]=matrix[0][0]-m.matrix[0][0];
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i]-m.matrix[0][i];
                        return new IntegerDiagonalMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this matrix by a scalar.
        * @param x an integer
        * @return an integer diagonal matrix
        */
        public IntegerMatrix scalarMultiply(final int x) {
                final int array[]=new int[matrix[0].length];
                array[0]=x*matrix[0][0];
                for(int i=1;i<array.length;i++)
                        array[i]=x*matrix[0][i];
                return new IntegerDiagonalMatrix(array);
        }

// MATRIX MULTIPLICATION

        /**
        * Returns the multiplication of a vector by this matrix.
        * @param v an integer vector
        * @exception DimensionException If the matrix and vector are incompatible.
        */
        public IntegerVector multiply(final IntegerVector v) {
                if(matrix[0].length==v.dimension()) {
                        final int array[]=new int[matrix[0].length];
                        array[0]=matrix[0][0]*v.getComponent(0);
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i]*v.getComponent(i);
                        return new IntegerVector(array);
                } else
                        throw new DimensionException("Matrix and vector are incompatible.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m an integer matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerMatrix multiply(final IntegerMatrix m) {
                if(m instanceof IntegerDiagonalMatrix)
                        return multiply((IntegerDiagonalMatrix)m);
                else if(m instanceof IntegerTridiagonalMatrix)
                        return rawMultiply((IntegerTridiagonalMatrix)m);
                else if(m instanceof IntegerSquareMatrix)
                        return rawMultiply((IntegerSquareMatrix)m);
                else
                        return rawMultiply(m);
        }
        private IntegerMatrix rawMultiply(final IntegerMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final int array[][]=new int[matrix[0].length][m.columns()];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[0][i]*m.getElement(i,0);
                                for(j=1;j<array[0].length;j++)
                                        array[i][j]=matrix[0][i]*m.getElement(i,j);
                        }
                        return new IntegerMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m an integer square matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerSquareMatrix multiply(final IntegerSquareMatrix m) {
                if(m instanceof IntegerDiagonalMatrix)
                        return multiply((IntegerDiagonalMatrix)m);
                else if(m instanceof IntegerTridiagonalMatrix)
                        return rawMultiply((IntegerTridiagonalMatrix)m);
                else
                        return rawMultiply(m);
        }
        private IntegerSquareMatrix rawMultiply(final IntegerSquareMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final int array[][]=new int[matrix[0].length][matrix[0].length];
                        for(int j,i=0;i<array.length;i++) {
                                array[i][0]=matrix[0][i]*m.getElement(i,0);
                                for(j=1;j<array.length;j++)
                                        array[i][j]=matrix[0][i]*m.getElement(i,j);
                        }
                        return new IntegerSquareMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m an integer tridiagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerSquareMatrix multiply(final IntegerTridiagonalMatrix m) {
                if(m instanceof IntegerDiagonalMatrix)
                        return multiply((IntegerDiagonalMatrix)m);
                else
                        return rawMultiply(m);
        }
        private IntegerSquareMatrix rawMultiply(final IntegerTridiagonalMatrix m) {
                int mRow=matrix[0].length;
                if(mRow==m.rows()) {
                        final IntegerTridiagonalMatrix ans=new IntegerTridiagonalMatrix(mRow);
                        ans.setElement(0,0,matrix[0][0]*m.getElement(0,0));
                        ans.setElement(0,1,matrix[0][0]*m.getElement(0,1));
                        mRow--;
                        for(int i=1;i<mRow;i++) {
                                ans.setElement(i,i-1,matrix[0][i]*m.getElement(i,i-1));
                                ans.setElement(i,i,matrix[0][i]*m.getElement(i,i));
                                ans.setElement(i,i+1,matrix[0][i]*m.getElement(i,i+1));
                        }
                        ans.setElement(mRow,mRow-1,matrix[0][mRow]*m.getElement(mRow,mRow-1));
                        ans.setElement(mRow,mRow,matrix[0][mRow]*m.getElement(mRow,mRow));
                        return ans;
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }
        /**
        * Returns the multiplication of this matrix and another.
        * @param m an integer diagonal matrix
        * @exception MatrixDimensionException If the matrices are different sizes.
        */
        public IntegerDiagonalMatrix multiply(final IntegerDiagonalMatrix m) {
                if(matrix[0].length==m.rows()) {
                        final int array[]=new int[matrix[0].length];
                        array[0]=matrix[0][0]*m.getElement(0,0);
                        for(int i=1;i<array.length;i++)
                                array[i]=matrix[0][i]*m.getElement(i,i);
                        return new IntegerDiagonalMatrix(array);
                } else
                        throw new MatrixDimensionException("Matrices are different sizes.");
        }

// TRANSPOSE

        /**
        * Returns the transpose of this matrix.
        * @return an integer diagonal matrix
        */
        public Matrix transpose() {
                return this;
        }

// LU DECOMPOSITION

        /**
        * Returns the LU decomposition of this matrix.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] luDecompose() {
                final DoubleDiagonalMatrix lu[]=new DoubleDiagonalMatrix[2];
                lu[0]=DoubleDiagonalMatrix.identity(matrix[0].length);
                lu[1]=(DoubleDiagonalMatrix)this.toDoubleMatrix();
                return lu;
        }

// CHOLESKY DECOMPOSITION

        /**
        * Returns the Cholesky decomposition of this matrix.
        * Matrix must be symmetric and positive definite.
        * @return an array with [0] containing the L-matrix and [1] containing the U-matrix.
        */
        public DoubleSquareMatrix[] choleskyDecompose() {
                final DoubleDiagonalMatrix lu[]=new DoubleDiagonalMatrix[2];
                final double array[]=new double[matrix[0].length];
                array[0]=Math.sqrt(matrix[0][0]);
                for(int i=1;i<array.length;i++)
                        array[i]=Math.sqrt(matrix[0][i]);
                lu[0]=new DoubleDiagonalMatrix(array);
                lu[1]=lu[0];
                return lu;
        }
}

