package JSci.maths;

import JSci.maths.groups.AbelianGroupMember;

/**
* The IntegerVector class encapsulates vectors containing integers.
* @version 2.1
* @author Mark Hale
*/
public class IntegerVector extends MathVector {
        /**
        * Storage format identifier.
        */
        protected final static int ARRAY_1D=1;
        protected final static int storageFormat=ARRAY_1D;
        /**
        * Array containing the components of the vector.
        */
        protected int vector[];
        /**
        * Constructs an empty vector.
        * @param dim the dimension of the vector.
        */
        public IntegerVector(final int dim) {
                vector=new int[dim];
        }
        /**
        * Constructs a vector by wrapping an array.
        * @param array an assigned value
        */
        public IntegerVector(final int array[]) {
                vector=array;
        }
        /**
        * Compares two integer vectors for equality.
        * @param a an integer vector
        */
        public boolean equals(Object a) {
                if(a!=null && (a instanceof IntegerVector) && vector.length==((IntegerVector)a).dimension()) {
                        final IntegerVector iv=(IntegerVector)a;
                        for(int i=0;i<vector.length;i++) {
                                if(vector[i]!=iv.getComponent(i))
                                        return false;
                        }
                        return true;
                } else
                        return false;
        }
        /**
        * Returns a comma delimited string representing the value of this vector.
        */
        public String toString() {
                final StringBuffer buf=new StringBuffer(vector.length);
                int i;
                for(i=0;i<vector.length-1;i++) {
                        buf.append(vector[i]);
                        buf.append(',');
                }
                buf.append(vector[i]);
                return buf.toString();
        }
        /**
        * Returns a hashcode for this vector.
        */
        public int hashCode() {
                return (int)Math.exp(norm());
        }
        /**
        * Converts this vector to a double vector.
        * @return a double vector
        */
        public DoubleVector toDoubleVector() {
                final double ans[]=new double[vector.length];
                for(int i=0;i<vector.length;i++)
                        ans[i]=vector[i];
                return new DoubleVector(ans);
        }
        /**
        * Converts this vector to a complex vector.
        * @return a complex vector
        */
        public ComplexVector toComplexVector() {
                final Complex ans[]=new Complex[vector.length];
                for(int i=0;i<vector.length;i++)
                        ans[i]=new Complex(vector[i],0.0);
                return new ComplexVector(ans);
        }
        /**
        * Returns a component of this vector.
        * @param n index of the vector component
        * @exception VectorDimensionException If attempting to access an invalid component.
        */
        public int getComponent(final int n) {
                if(n>=0 && n<vector.length)
                        return vector[n];
                else
                        throw new VectorDimensionException("Invalid component.");
        }
        /**
        * Sets the value of a component of this vector.
        * @param n index of the vector component
        * @param x an integer
        * @exception VectorDimensionException If attempting to access an invalid component.
        */
        public void setComponent(final int n, final int x) {
                if(n>=0 && n<vector.length)
                        vector[n]=x;
                else
                        throw new VectorDimensionException("Invalid component.");
        }
        /**
        * Returns the vector's dimension.
        */
        public int dimension() {
                return vector.length;
        }
        /**
        * Returns the ln-norm.
        */
        public double norm(final int n) {
                double answer=Math.pow(vector[0],n);
                for(int i=1;i<vector.length;i++)
                        answer+=Math.pow(vector[i],n);
                return Math.pow(answer,1.0/n);
        }
        /**
        * Returns the l2-norm (magnitude).
        */
        public double norm() {
                int answer=vector[0]*vector[0];
                for(int i=1;i<vector.length;i++)
                        answer+=vector[i]*vector[i];
                return Math.sqrt(answer);
        }
        /**
        * Returns the l(infinity)-norm.
        * @author Taber Smith
        */
        public double infNorm() {
                int infNorm=vector[0];
                for(int i=1;i<vector.length;i++) {
                        if(vector[i]>infNorm)
                                infNorm=vector[i];
                }
                return infNorm;
        }

//============
// OPERATIONS
//============

        /**
        * Returns the negative of this vector.
        */
        public AbelianGroupMember negate() {
                final int array[]=new int[vector.length];
                array[0]=-vector[0];
                for(int i=1;i<array.length;i++)
                        array[i]=-vector[i];
                return new IntegerVector(array);
        }

// ADDITION

        /**
        * Returns the addition of this vector and another.
        */
        public AbelianGroupMember add(final AbelianGroupMember v) {
                if(v instanceof IntegerVector)
                        return add((IntegerVector)v);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the addition of this vector and another.
        * @param v an integer vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public IntegerVector add(final IntegerVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawAdd(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        final int array[]=new int[vector.length];
                                        array[0]=vector[0]+v.getComponent(0);
                                        for(int i=1;i<array.length;i++)
                                                array[i]=vector[i]+v.getComponent(i);
                                        return new IntegerVector(array);
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private IntegerVector rawAdd(final IntegerVector v) {
                if(vector.length==v.vector.length) {
                        final int array[]=new int[vector.length];
                        array[0]=vector[0]+v.vector[0];
                        for(int i=1;i<array.length;i++)
                                array[i]=vector[i]+v.vector[i];
                        return new IntegerVector(array);
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }

// SUBTRACTION

        /**
        * Returns the subtraction of this vector by another.
        */
        public AbelianGroupMember subtract(final AbelianGroupMember v) {
                if(v instanceof IntegerVector)
                        return subtract((IntegerVector)v);
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the subtraction of this vector by another.
        * @param v an integer vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public IntegerVector subtract(final IntegerVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawSubtract(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        final int array[]=new int[vector.length];
                                        array[0]=vector[0]-v.getComponent(0);
                                        for(int i=1;i<array.length;i++)
                                                array[i]=vector[i]-v.getComponent(i);
                                        return new IntegerVector(array);
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private IntegerVector rawSubtract(final IntegerVector v) {
                if(vector.length==v.vector.length) {
                        final int array[]=new int[vector.length];
                        array[0]=vector[0]-v.vector[0];
                        for(int i=1;i<array.length;i++)
                                array[i]=vector[i]-v.vector[i];
                        return new IntegerVector(array);
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }

// SCALAR MULTIPLICATION

        /**
        * Returns the multiplication of this vector by a scalar.
        */
        public ModuleMember scalarMultiply(RingMember x) {
                if(x instanceof MathInteger)
                        return scalarMultiply(((MathInteger)x).value());
                else
                        throw new IllegalArgumentException("Member class not recognised by this method.");
        }
        /**
        * Returns the multiplication of this vector by a scalar.
        * @param x an integer
        */
        public IntegerVector scalarMultiply(final int x) {
                final int array[]=new int[vector.length];
                array[0]=x*vector[0];
                for(int i=1;i<array.length;i++)
                        array[i]=x*vector[i];
                return new IntegerVector(array);
        }

// SCALAR PRODUCT

        /**
        * Returns the scalar product of this vector and another.
        * @param v an integer vector
        * @exception VectorDimensionException If the vectors are different sizes.
        */
        public int scalarProduct(final IntegerVector v) {
                switch(v.storageFormat) {
                        case ARRAY_1D: return rawScalarProduct(v);
                        default: 
                                if(vector.length==v.dimension()) {
                                        int answer=vector[0]*v.getComponent(0);
                                        for(int i=1;i<vector.length;i++)
                                                answer+=vector[i]*v.getComponent(i);
                                        return answer;
                                } else
                                        throw new VectorDimensionException("Vectors are different sizes.");
                }
        }
        private int rawScalarProduct(final IntegerVector v) {
                if(vector.length==v.vector.length) {
                        int answer=vector[0]*v.vector[0];
                        for(int i=1;i<vector.length;i++)
                                answer+=vector[i]*v.vector[i];
                        return answer;
                } else
                        throw new VectorDimensionException("Vectors are different sizes.");
        }
}

