package JSci.maths.chaos;

import JSci.maths.*;

/**
* The HenonMap class provides an object that encapsulates the Henon map.
* @version 1.0
* @author Mark Hale
*/
public final class HenonMap extends Object implements NMapping {
        private Complex a;
        private Complex b;
        /**
        * Chaotic a parameter value.
        */
        public final static Complex A_CHAOS=new Complex(1.4,0.0);
        /**
        * Chaotic b parameter value.
        */
        public final static Complex B_CHAOS=new Complex(0.3,0.0);
        /**
        * Constructs a Henon map.
        * @param aval the value of the a parameter
        * @param bval the value of the b parameter
        */
        public HenonMap(double aval,double bval) {
                a=new Complex(aval,0.0);
                b=new Complex(bval,0.0);
        }
        /**
        * Constructs a Henon map.
        * @param aval the value of the a parameter
        * @param bval the value of the b parameter
        */
        public HenonMap(Complex aval,Complex bval) {
                a=aval;
                b=bval;
        }
        /**
        * Performs the mapping.
        * @param x a 2-D double array
        * @return a 2-D double array
        */
        public double[] map(double x[]) {
                double ans[]=new double[2];
                ans[0]=1-a.real()*x[0]*x[0]+x[1];
                ans[1]=b.real()*x[0];
                return ans;
        }
        /**
        * Performs the mapping.
        * @param z a 2-D complex array
        * @return a 2-D complex array
        */
        public Complex[] map(Complex z[]) {
                Complex ans[]=new Complex[2];
                ans[0]=Complex.ONE.add(z[1]).subtract(a.multiply(z[0].sqr()));
                ans[1]=b.multiply(z[0]);
                return ans;
        }
        /**
        * Iterates the map.
        * @param n the number of iterations
        * @param x the initial values (2-D)
        * @return a 2-D double array
        */
        public double[] iterate(int n,double x[]) {
                double xn[]=map(x);
                for(int i=1;i<n;i++)
                        xn=map(xn);
                return xn;
        }
        /**
        * Iterates the map.
        * @param n the number of iterations
        * @param z the initial values (2-D)
        * @return a 2-D complex array
        */
        public Complex[] iterate(int n,Complex z[]) {
                Complex zn[]=map(z);
                for(int i=1;i<n;i++)
                        zn=map(zn);
                return zn;
        }
}

