package JSci.swing;

import java.awt.*;
import JSci.awt.*;

/**
* The JGraph2D superclass provides an abstract encapsulation of 2D graphs.
* @version 1.0
* @author Mark Hale
*/
public abstract class JGraph2D extends JDoubleBufferedComponent implements GraphDataListener {
        /**
        * Data model.
        */
        protected Graph2DModel model;
        /**
        * Axis numbering.
        */
        private boolean numbering=true;
        /**
        * Origin.
        */
        protected Point origin;
        /**
        * Min and max data points.
        */
        protected float minX,minY,maxX,maxY;
        /**
        * Axis scaling.
        */
        protected float xScale,yScale;
        /**
        * Padding.
        */
        protected final int scalePad=5,axisPad=25;
        /**
        * Constructs a 2D graph.
        */
        public JGraph2D(Graph2DModel gm) {
                model=gm;
                model.addGraphDataListener(this);
        }
        /**
        * Sets the data plotted by this graph to the specified data.
        */
        public final void setModel(Graph2DModel gm) {
                model.removeGraphDataListener(this);
                model=gm;
                model.addGraphDataListener(this);
                dataChanged(new GraphDataEvent(model));
        }
        /**
        * Returns the model used by this graph.
        */
        public final Graph2DModel getModel() {
                return model;
        }
        /**
        * Turns axis numbering on/off.
        */
        public final void setNumbering(boolean flag) {
                numbering=flag;
        }
        /**
        * Reshapes the JGraph2D to the specified bounding box.
        */
        public final void setBounds(int x,int y,int width,int height) {
                super.setBounds(x,y,width,height);
                rescale();
        }
        /**
        * Rescales the JGraph2D.
        */
        protected final void rescale() {
                final Dimension s=getMinimumSize();
                final int thisWidth=Math.max(getWidth(),s.width);
                final int thisHeight=Math.max(getHeight(),s.height);
                xScale=(thisWidth-2*(axisPad+scalePad))/(maxX-minX);
                yScale=(thisHeight-2*(axisPad+scalePad))/(maxY-minY);
                origin=new Point(Math.round(-minX*xScale)+axisPad+scalePad,Math.round(maxY*yScale)+axisPad+scalePad);
                redraw();
        }
        /**
        * Converts a data point to screen coordinates.
        */
        protected final Point dataToScreen(float x,float y) {
                return new Point(origin.x+Math.round(xScale*x),origin.y-Math.round(yScale*y));
        }
        /**
        * Draws the graph axes.
        */
        protected final void drawAxes(Graphics g) {
// axis
                g.setColor(Color.black);
                g.drawLine(axisPad,origin.y,getWidth()-axisPad,origin.y);
                g.drawLine(origin.x,axisPad,origin.x,getHeight()-axisPad);
// numbering
                if(numbering) {
                        String str;
                        int strWidth;
                        final int strHeight=g.getFontMetrics().getHeight();
                        Point p;
// x-axis numbering
                        float x;
                        final float dx=round(40.0f/xScale);
                        for(x=dx;x<=maxX;x+=dx) {
                                str=String.valueOf(round(x));
// add a + prefix to compensate for - prefix in negative number strings when calculating length
                                strWidth=g.getFontMetrics().stringWidth('+'+str);
                                p=dataToScreen(x,0.0f);
                                g.drawLine(p.x,p.y,p.x,p.y+5);
                                g.drawString(str,p.x-strWidth/2,origin.y+strHeight+5);
                        }
                        for(x=-dx;x>=minX;x-=dx) {
                                str=String.valueOf(round(x));
                                strWidth=g.getFontMetrics().stringWidth(str);
                                p=dataToScreen(x,0.0f);
                                g.drawLine(p.x,p.y,p.x,p.y+5);
                                g.drawString(str,p.x-strWidth/2,origin.y+strHeight+5);
                        }
// y-axis numbering
                        float y;
                        final float dy=round(40.0f/yScale);
                        for(y=dy;y<=maxY;y+=dy) {
                                str=String.valueOf(round(y));
                                strWidth=g.getFontMetrics().stringWidth(str);
                                p=dataToScreen(0.0f,y);
                                g.drawLine(p.x,p.y,p.x-5,p.y);
                                g.drawString(str,origin.x-strWidth-8,p.y+strHeight/4);
                        }
                        for(y=-dy;y>=minY;y-=dy) {
                                str=String.valueOf(round(y));
                                strWidth=g.getFontMetrics().stringWidth(str);
                                p=dataToScreen(0.0f,y);
                                g.drawLine(p.x,p.y,p.x-5,p.y);
                                g.drawString(str,origin.x-strWidth-8,p.y+strHeight/4);
                        }
                }
        }
        /**
        * Rounds numbers to so many significant figures.
        */
        protected final float round(float x) {
                final int SIG_FIG=2;
                int sign=1;
                if(x<0.0f)
                        sign=-1;
                float mag=Math.abs(x);
                int places;
                float tmp,factor;
                if(mag<1.0f) {
                        tmp=10.0f*mag;
                        for(places=1;tmp<1.0f;places++)
                                tmp*=10.0f;
                        factor=(float)Math.pow(10.0f,places+SIG_FIG-1);
                } else {
                        tmp=mag/10.0f;
                        for(places=1;tmp>1.0f;places++)
                                tmp/=10.0f;
                        factor=(float)Math.pow(10.0f,SIG_FIG-places);
                }
                return (float)(sign*Math.rint(mag*factor)/factor);
        }
}

