package JSci.swing;

import java.awt.*;
import java.awt.event.*;
import JSci.awt.*;

/**
* A line trace Swing component.
* @version 1.0
* @author Mark Hale
*/
public final class JLineTrace extends JDoubleBufferedComponent {
        /**
        * Data points.
        */
        private float data[][];
        /**
        * Data cursor.
        */
        private int dataCursor;
        /**
        * Data resolution.
        */
        private float dataRes;
        /**
        * Axis numbering.
        */
        private boolean numbering=true;
        /**
        * Origin.
        */
        private Point origin;
        /**
        * Min and max data points.
        */
        private float minX,minY,maxX,maxY;
        /**
        * Axis scaling.
        */
        private float xScale,yScale;
        /**
        * Padding.
        */
        private final int scalePad=5,axisPad=25;
        /**
        * Constructs a line trace.
        */
        public JLineTrace(float minx,float maxx,float miny,float maxy) {
                addMouseMotionListener(new MouseLineAdapter());
                setXExtrema(minx,maxx);
                setYExtrema(miny,maxy);
                setSampleNumber(50);
        }
        /**
        * Gets the data sampled by this JLineTrace.
        */
        public Graph2DModel getModel() {
                DefaultGraph2DModel model=new DefaultGraph2DModel();
                model.setXAxis(data[0]);
                model.addSeries(data[1]);
                return model;
        }
        /**
        * Turns axis numbering on/off.
        */
        public void setNumbering(boolean flag) {
                numbering=flag;
        }
        /**
        * Sets the minimum/maximum values on the x-axis.
        */
        public void setXExtrema(float min,float max) {
                if(max<min)
                        throw new IllegalArgumentException("Maximum should be greater than minimum; max = "+max+" and min = "+min);
                minX=min;
                maxX=max;
                rescale();
        }
        /**
        * Sets the minimum/maximum values on the y-axis.
        */
        public void setYExtrema(float min,float max) {
                if(max<min)
                        throw new IllegalArgumentException("Maximum should be greater than minimum; max = "+max+" and min = "+min);
                minY=min;
                maxY=max;
                rescale();
        }
        /**
        * Sets the number of samples.
        */
        public void setSampleNumber(int n) {
                data=new float[2][n];
                dataRes=(maxX-minX)/n;
        }
        /**
        * Clear the trace.
        */
        public void clear() {
                dataCursor=0;
                redraw();
        }
        /**
        * Reshapes the JLineTrace to the specified bounding box.
        */
        public final void setBounds(int x,int y,int width,int height) {
                super.setBounds(x,y,width,height);
                rescale();
        }
        /**
        * Rescales the JLineTrace.
        */
        private void rescale() {
                final Dimension s=getMinimumSize();
                final int thisWidth=Math.max(getWidth(),s.width);
                final int thisHeight=Math.max(getHeight(),s.height);
                xScale=(thisWidth-2*(axisPad+scalePad))/(maxX-minX);
                yScale=(thisHeight-2*(axisPad+scalePad))/(maxY-minY);
                origin=new Point(Math.round(-minX*xScale)+axisPad+scalePad,Math.round(maxY*yScale)+axisPad+scalePad);
                redraw();
        }
        /**
        * Returns the preferred size of this component.
        */
        public Dimension getPreferredSize() {
                return getMinimumSize();
        }
        /**
        * Returns the minimum size of this component.
        */
        public Dimension getMinimumSize() {
                return new Dimension(250,250);
        }
        /**
        * Converts a screen point to data coordinates.
        */
        private float[] screenToData(Point p) {
                float xy[]={(p.x-origin.x)/xScale,(origin.y-p.y)/yScale};
                return xy;
        }
        /**
        * Converts a data point to screen coordinates.
        */
        private Point dataToScreen(float x,float y) {
                return new Point(origin.x+Math.round(xScale*x),origin.y-Math.round(yScale*y));
        }
        /**
        * Paint the trace.
        */
        protected void offscreenPaint(Graphics g) {
                drawAxes(g);
// lines
                Point p1,p2;
                p1=dataToScreen(data[0][0],data[1][0]);
                for(int i=1;i<dataCursor;i++) {
                        p2=dataToScreen(data[0][i],data[1][i]);
                        g.drawLine(p1.x,p1.y,p2.x,p2.y);
                        p1=p2;
                }
        }
        /**
        * Draws the graph axes.
        */
        private void drawAxes(Graphics g) {
// axis
                g.setColor(Color.black);
                g.drawLine(axisPad,origin.y,getWidth()-axisPad,origin.y);
                g.drawLine(origin.x,axisPad,origin.x,getHeight()-axisPad);
// numbering
                if(numbering) {
                        String str;
                        int strWidth;
                        final int strHeight=g.getFontMetrics().getHeight();
                        Point p;
// x-axis numbering
                        float x;
                        final float dx=round(40.0f/xScale);
                        for(x=dx;x<=maxX;x+=dx) {
                                str=String.valueOf(round(x));
// add a + prefix to compensate for - prefix in negative number strings when calculating length
                                strWidth=g.getFontMetrics().stringWidth('+'+str);
                                p=dataToScreen(x,0.0f);
                                g.drawLine(p.x,p.y,p.x,p.y+5);
                                g.drawString(str,p.x-strWidth/2,origin.y+strHeight+5);
                        }
                        for(x=-dx;x>=minX;x-=dx) {
                                str=String.valueOf(round(x));
                                strWidth=g.getFontMetrics().stringWidth(str);
                                p=dataToScreen(x,0.0f);
                                g.drawLine(p.x,p.y,p.x,p.y+5);
                                g.drawString(str,p.x-strWidth/2,origin.y+strHeight+5);
                        }
// y-axis numbering
                        float y;
                        final float dy=round(40.0f/yScale);
                        for(y=dy;y<=maxY;y+=dy) {
                                str=String.valueOf(round(y));
                                strWidth=g.getFontMetrics().stringWidth(str);
                                p=dataToScreen(0.0f,y);
                                g.drawLine(p.x,p.y,p.x-5,p.y);
                                g.drawString(str,origin.x-strWidth-8,p.y+strHeight/4);
                        }
                        for(y=-dy;y>=minY;y-=dy) {
                                str=String.valueOf(round(y));
                                strWidth=g.getFontMetrics().stringWidth(str);
                                p=dataToScreen(0.0f,y);
                                g.drawLine(p.x,p.y,p.x-5,p.y);
                                g.drawString(str,origin.x-strWidth-8,p.y+strHeight/4);
                        }
                }
        }
        /**
        * Rounds numbers to so many significant figures.
        */
        private float round(float x) {
                final int SIG_FIG=2;
                int sign=1;
                if(x<0.0f)
                        sign=-1;
                float mag=Math.abs(x);
                int places;
                float tmp,factor;
                if(mag<1.0f) {
                        tmp=10.0f*mag;
                        for(places=1;tmp<1.0f;places++)
                                tmp*=10.0f;
                        factor=(float)Math.pow(10.0f,places+SIG_FIG-1);
                } else {
                        tmp=mag/10.0f;
                        for(places=1;tmp>1.0f;places++)
                                tmp/=10.0f;
                        factor=(float)Math.pow(10.0f,SIG_FIG-places);
                }
                return (float)(sign*Math.rint(mag*factor)/factor);
        }
        class MouseLineAdapter extends MouseMotionAdapter {
                public void mouseDragged(MouseEvent evt) {
                        float xy[]=screenToData(evt.getPoint());
                        if(xy[0]>=dataCursor*dataRes+minX && dataCursor<data[0].length) {
                                data[0][dataCursor]=xy[0];
                                data[1][dataCursor]=xy[1];
                                dataCursor++;
                        }
                        redraw();
                }
        }
}

